/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.aws2.translate;

import org.apache.camel.EndpointInject;
import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.test.spring.junit5.CamelSpringTestSupport;
import org.junit.jupiter.api.Test;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import software.amazon.awssdk.services.translate.model.TranslateTextRequest;

import static org.junit.jupiter.api.Assertions.assertEquals;

public class Translate2ProducerSpringTest extends CamelSpringTestSupport {

    @EndpointInject("mock:result")
    private MockEndpoint mock;

    @Test
    public void translateTextTest() throws Exception {

        mock.expectedMessageCount(1);
        Exchange exchange = template.request("direct:translateText", new Processor() {
            @Override
            public void process(Exchange exchange) throws Exception {
                exchange.getIn().setHeader(Translate2Constants.OPERATION, Translate2Operations.translateText);
                exchange.getIn().setHeader(Translate2Constants.SOURCE_LANGUAGE, Translate2LanguageEnum.ITALIAN);
                exchange.getIn().setHeader(Translate2Constants.TARGET_LANGUAGE, Translate2LanguageEnum.ENGLISH);
                exchange.getIn().setBody("ciao");
            }
        });

        assertMockEndpointsSatisfied();

        String resultGet = exchange.getIn().getBody(String.class);
        assertEquals("Hello", resultGet);

    }

    @Test
    public void translateTextPojoTest() throws Exception {

        mock.expectedMessageCount(1);
        Exchange exchange = template.request("direct:translatePojoText", new Processor() {
            @Override
            public void process(Exchange exchange) throws Exception {
                exchange.getIn()
                        .setBody(TranslateTextRequest.builder().sourceLanguageCode(Translate2LanguageEnum.ITALIAN.toString())
                                .targetLanguageCode(Translate2LanguageEnum.ENGLISH.toString()).text("ciao").build());
            }
        });

        assertMockEndpointsSatisfied();

        String resultGet = exchange.getIn().getBody(String.class);
        assertEquals("Hello", resultGet);

    }

    @Test
    public void translateTextTestOptions() throws Exception {

        mock.expectedMessageCount(1);
        Exchange exchange = template.request("direct:translateTextOptions", new Processor() {
            @Override
            public void process(Exchange exchange) throws Exception {
                exchange.getIn().setBody("ciao");
            }
        });

        assertMockEndpointsSatisfied();

        String resultGet = exchange.getIn().getBody(String.class);
        assertEquals("Hello", resultGet);

    }

    @Override
    protected ClassPathXmlApplicationContext createApplicationContext() {
        return new ClassPathXmlApplicationContext(
                "org/apache/camel/component/aws2/translate/Translate2ComponentSpringTest-context.xml");
    }
}
