/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.google.calendar;

import java.io.IOException;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import com.google.api.services.calendar.model.Calendar;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelExecutionException;
import org.apache.camel.support.PropertyBindingSupport;
import org.apache.camel.test.junit5.CamelTestSupport;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.TestInstance;

/**
 * Abstract base class for GoogleCalendar Integration tests generated by Camel API component maven plugin.
 */
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
public class AbstractGoogleCalendarTestSupport extends CamelTestSupport {

    private static final String TEST_OPTIONS_PROPERTIES = "/test-options.properties";
    private static Properties properties;
    private Calendar calendar;

    static {
        properties = loadProperties();
    }

    private static Properties loadProperties() {
        // read GoogleCalendar component configuration from
        // TEST_OPTIONS_PROPERTIES
        final Properties properties = new Properties();
        try {
            properties.load(AbstractGoogleCalendarTestSupport.class.getResourceAsStream(TEST_OPTIONS_PROPERTIES));
        } catch (Exception e) {
            throw new RuntimeException(String.format("%s could not be loaded: %s", TEST_OPTIONS_PROPERTIES, e.getMessage()), e);
        }

        return properties;
    }

    // Used by JUnit to determine whether or not to run the integration tests
    @SuppressWarnings("unused")
    private static boolean hasCredentials() {
        loadProperties();

        return (!properties.getProperty("clientId", "").isEmpty())
                && (!properties.getProperty("clientSecret", "").isEmpty())
                && (!properties.getProperty("accessToken", "").isEmpty());
    }

    @BeforeEach
    public void createTestCalendar() {
        Calendar calendar = new Calendar();
        SecureRandom rand = new SecureRandom();

        calendar.setSummary("camel-calendar-" + rand.nextInt());
        calendar.setTimeZone("America/St_Johns");

        this.calendar = requestBody("google-calendar://calendars/insert?inBody=content", calendar);
    }

    @AfterEach
    public void deleteTestCalendar() {
        try {
            if (calendar != null) {
                requestBody("google-calendar://calendars/delete?inBody=calendarId", getCalendar().getId());
                setCalendar(null);
            }
        } catch (Exception e) {
        }
    }

    @Override
    protected CamelContext createCamelContext() throws Exception {

        final CamelContext context = super.createCamelContext();

        Map<String, Object> options = getTestOptions();

        final GoogleCalendarConfiguration configuration = new GoogleCalendarConfiguration();
        PropertyBindingSupport.bindProperties(context, configuration, options);

        // add GoogleCalendarComponent to Camel context
        final GoogleCalendarComponent component = new GoogleCalendarComponent(context);
        component.setConfiguration(configuration);
        context.addComponent("google-calendar", component);

        return context;
    }

    private Map<String, Object> getTestOptions() throws IOException {
        Map<String, Object> options = new HashMap<>();
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            options.put(entry.getKey().toString(), entry.getValue());
        }
        return options;
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBodyAndHeaders(String endpointUri, Object body, Map<String, Object> headers)
            throws CamelExecutionException {
        return (T) template().requestBodyAndHeaders(endpointUri, body, headers);
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBody(String endpoint, Object body) throws CamelExecutionException {
        return (T) template().requestBody(endpoint, body);
    }

    public Calendar getCalendar() {
        if (calendar == null) {
            createTestCalendar();
        }
        return calendar;
    }

    public void setCalendar(Calendar calendar) {
        this.calendar = calendar;
    }
}
