/*
 * Copyright 2009 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.netty.handler.codec.protobuf;

import org.jboss.netty.buffer.ChannelBuffer;
import org.jboss.netty.channel.Channel;
import org.jboss.netty.channel.ChannelHandlerContext;
import org.jboss.netty.handler.codec.frame.FrameDecoder;

import com.google.protobuf.CodedInputStream;

/**
 * A decoder that splits the received {@link ChannelBuffer}s dynamically by the
 * value of the Google Protocol Buffers
 * <a href="http://code.google.com/apis/protocolbuffers/docs/encoding.html#varints">Base
 * 128 Varints</a> integer length field in the message.  For example:
 * <pre>
 * BEFORE DECODE (302 bytes)       AFTER DECODE (300 bytes)
 * +--------+---------------+      +---------------+
 * | Length | Protobuf Data |----->| Protobuf Data |
 * | 0xAC02 |  (300 bytes)  |      |  (300 bytes)  |
 * +--------+---------------+      +---------------+
 * </pre>
 *
 * @see com.google.protobuf.CodedInputStream
 *
 * @author <a href="http://www.jboss.org/netty/">The Netty Project</a>
 * @author Tomasz Blachowicz (tblachowicz@gmail.com)
 * @author <a href="http://gleamynode.net/">Trustin Lee</a>
 *
 * @version $Rev: 2316 $, $Date: 2010-06-23 01:22:57 -0400 (Wed, 23 Jun 2010) $
 */
public class ProtobufVarint32FrameDecoder extends FrameDecoder {

    // TODO maxFrameLength + safe skip (just like LengthFieldBasedFrameDecoder)

    /**
     * Creates a new instance.
     */
    public ProtobufVarint32FrameDecoder() {
        super();
    }

    @Override
    protected Object decode(ChannelHandlerContext ctx, Channel channel, ChannelBuffer buffer) throws Exception {
        buffer.markReaderIndex();
        byte[] buf = new byte[5];
        for (int i = 0; i < 5; i ++) {
            if (!buffer.readable()) {
                break;
            }

            buf[i] = buffer.readByte();
            if (buf[i] >= 0) {
                int messageSize = CodedInputStream.newInstance(buf, 0, i + 1).readRawVarint32();
                if (buffer.readableBytes() < messageSize) {
                    break;
                }

                return buffer.readBytes(messageSize);
            }
        }

        buffer.resetReaderIndex();
        return null;
    }
}
