/*
 * ModeShape (http://www.modeshape.org)
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * See the AUTHORS.txt file in the distribution for a full listing of 
 * individual contributors.
 *
 * ModeShape is free software. Unless otherwise indicated, all code in ModeShape
 * is licensed to you under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * ModeShape is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.modeshape.common.util;

import java.io.InputStream;

/**
 * Utility to search & load resources from the CP.
 *
 * @author Horia Chiorean (hchiorea@redhat.com)
 */
public final class ResourceLookup {

    private ResourceLookup() {
    }

    /**
     * Returns the stream of a resource at a given path, using some optional class loaders.
     *
     * @param path the path to search
     * @param classLoader a {@link java.lang.ClassLoader} instance to use when searching; may be null.
     * @param useTLCL {@code true} if the thread local class loader should be used as well, in addition to {@code classLoader}
     * @return a {@link java.io.InputStream} if the resource was found, or {@code null} otherwise
     */
    public static InputStream read( String path, ClassLoader classLoader, boolean useTLCL ) {
        if (useTLCL) {
            InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(path);
            if (stream != null) {
                return stream;
            }
        }
        return classLoader != null ? classLoader.getResourceAsStream(path) : ResourceLookup.class.getResourceAsStream(path);
    }

    /**
     * Returns the stream of a resource at a given path, using the CL of a class.
     *
     * @param path the path to search
     * @param clazz a {@link java.lang.Class} instance which class loader should be used when doing the lookup.
     * @param useTLCL {@code true} if the thread local class loader should be used as well, in addition to {@code classLoader}
     * @return a {@link java.io.InputStream} if the resource was found, or {@code null} otherwise
     * @see org.modeshape.common.util.ResourceLookup#read(String, ClassLoader, boolean)
     */
    public static InputStream read( String path, Class<?> clazz, boolean useTLCL ) {
        return read(path, clazz.getClassLoader(), useTLCL);
    }
}
