"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.DevWorkspaceTasks = void 0;
const tslib_1 = require("tslib");
const fs = require("fs-extra");
const yaml = require("js-yaml");
const path = require("path");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
const openshift_1 = require("../../api/openshift");
const common_tasks_1 = require("../installers/common-tasks");
/**
 * Handle setup of the dev workspace operator controller.
 */
class DevWorkspaceTasks {
    constructor(flags) {
        this.flags = flags;
        this.devWorkspaceServiceAccount = 'devworkspace-controller';
        this.devWorkspaceRole = 'devworkspace-controller';
        this.devWorkspaceEditWorkspaceClusterRole = 'edit-workspaces';
        this.devWorkspaceViewWorkspaceClusterRole = 'view-workspaces';
        this.devWorkspaceRoleBinding = 'devworkspace-controller';
        this.deploymentName = 'devworkspace-controller';
        this.kubeHelper = new kube_1.KubeHelper(flags);
        this.cheHelper = new che_1.CheHelper(flags);
        this.openShiftHelper = new openshift_1.OpenShiftHelper();
    }
    getTemplatePath() {
        return path.join(this.flags.templates, 'devworkspace');
    }
    getNamespace() {
        return this.flags['dev-workspace-controller-namespace'];
    }
    /**
     * Returns list of tasks which setup dev-workspace.
     */
    getInstallTasks(flags) {
        return [
            common_tasks_1.createNamespaceTask(this.getNamespace(), {}),
            {
                title: `Create ServiceAccount ${this.devWorkspaceServiceAccount} in namespace ${this.getNamespace()}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield this.kubeHelper.serviceAccountExist(this.devWorkspaceServiceAccount, this.getNamespace());
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const serviceAccountPath = path.join(this.getTemplatePath(), 'service_account.yaml');
                        yield this.kubeHelper.createServiceAccountFromFile(serviceAccountPath, this.getNamespace());
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create Cluster Role ${this.devWorkspaceRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield this.kubeHelper.clusterRoleExist(this.devWorkspaceRole);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const rolePath = path.join(this.getTemplatePath(), 'role.yaml');
                        yield this.kubeHelper.createClusterRoleFromFile(rolePath, this.devWorkspaceRole);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRole ${this.devWorkspaceEditWorkspaceClusterRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield this.kubeHelper.clusterRoleExist(this.devWorkspaceEditWorkspaceClusterRole);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const clusterRolePath = path.join(this.getTemplatePath(), 'edit-workspaces-cluster-role.yaml');
                        yield this.kubeHelper.createClusterRoleFromFile(clusterRolePath, this.devWorkspaceEditWorkspaceClusterRole);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRole ${this.devWorkspaceViewWorkspaceClusterRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield this.kubeHelper.clusterRoleExist(this.devWorkspaceViewWorkspaceClusterRole);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const clusterRolePath = path.join(this.getTemplatePath(), 'view-workspaces-cluster-role.yaml');
                        yield this.kubeHelper.createClusterRoleFromFile(clusterRolePath, this.devWorkspaceViewWorkspaceClusterRole);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRoleBinding ${this.devWorkspaceRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield this.kubeHelper.clusterRoleBindingExist(this.devWorkspaceRoleBinding);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const roleBindingPath = path.join(this.getTemplatePath(), 'role_binding.yaml');
                        const rawYaml = yield fs.readFile(roleBindingPath, 'utf-8');
                        const clusterRoleBindingYaml = yaml.safeLoad(rawYaml);
                        clusterRoleBindingYaml.subjects[0].namespace = this.getNamespace();
                        yield this.kubeHelper.createClusterRoleBindingFrom(clusterRoleBindingYaml);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Create dev workspace CRDs',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exists = yield this.kubeHelper.crdExist('devworkspaces.workspace.devfile.io');
                    if (exists) {
                        task.title = `${task.title}...It already exists.`;
                        return;
                    }
                    const devfileApiCrdsPath = path.join(flags.templates, 'devfile-api', 'crds');
                    const devfileApiCrdsFiles = (yield fs.readdir(devfileApiCrdsPath)).map(file => path.join(devfileApiCrdsPath, file));
                    const devWorkspaceCrdsPath = path.join(this.getTemplatePath(), 'crds');
                    const devWorkspaceCrdsFiles = (yield fs.readdir(devWorkspaceCrdsPath)).map(file => path.join(devWorkspaceCrdsPath, file));
                    const crdFiles = [...devfileApiCrdsFiles, ...devWorkspaceCrdsFiles];
                    yield Promise.all(crdFiles.map((file) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        if (file.endsWith('.yaml')) {
                            return this.kubeHelper.createCrdFromFile(file);
                        }
                    })));
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Create dev workspace controller ConfigMap',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const yamlConfigFile = path.join(this.getTemplatePath(), 'controller_config.yaml');
                    const rawYaml = yield fs.readFile(yamlConfigFile, 'utf-8');
                    const configMapYaml = yaml.safeLoad(rawYaml);
                    const configMapName = 'devworkspace-controller';
                    const pluginRegistryURL = yield this.cheHelper.chePluginRegistryURL(flags.chenamespace);
                    configMapYaml.data['controller.plugin_registry.url'] = pluginRegistryURL;
                    if (flags.domain) {
                        configMapYaml.data['devworkspace.routing.cluster_host_suffix'] = flags.domain;
                    }
                    let webHooksValue = 'false';
                    let routingClass = 'basic';
                    if (ctx.isOpenShift) {
                        routingClass = 'openshift-oauth';
                        webHooksValue = 'true';
                    }
                    configMapYaml.data['controller.webhooks.enabled'] = webHooksValue;
                    configMapYaml.data['devworkspace.default_routing_class'] = routingClass;
                    configMapYaml.data['tls.insecure_skip_verify'] = 'true';
                    const configMap = yield this.kubeHelper.getConfigMap(configMapName, this.getNamespace());
                    if (configMap) {
                        task.title = `${task.title}...Already Exists. Replacing`;
                        yield this.kubeHelper.replaceNamespacedConfigMap(configMapName, this.getNamespace(), configMapYaml);
                    }
                    else {
                        yield this.kubeHelper.createNamespacedConfigMap(this.getNamespace(), configMapYaml);
                    }
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Create dev workspace controller',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exists = yield this.kubeHelper.deploymentExist('devworkspace-controller', this.getNamespace());
                    if (exists) {
                        task.title = `${task.title}...It already exists.`;
                        return;
                    }
                    const yamls = [];
                    if (ctx.isOpenShift) {
                        const yamlControllerFile = path.join(this.getTemplatePath(), 'os', 'controller.yaml');
                        const rawYaml = yield fs.readFile(yamlControllerFile, 'utf-8');
                        yaml.safeLoadAll(rawYaml, yaml => {
                            yamls.push(yaml);
                        });
                    }
                    else {
                        const yamlControllerFile = path.join(this.getTemplatePath(), 'k8s', 'controller.yaml');
                        const rawYaml = yield fs.readFile(yamlControllerFile, 'utf-8');
                        yaml.safeLoadAll(rawYaml, yaml => {
                            yamls.push(yaml);
                        });
                    }
                    yield Promise.all(yamls.map((yaml) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        if (yaml.kind === 'Deployment') {
                            // customize devworkspace controller image:
                            yaml.spec.template.spec.containers[0].image = flags['dev-workspace-controller-image'];
                            return this.kubeHelper.createDeploymentFrom(yaml, this.getNamespace());
                        }
                        if (yaml.kind === 'Service') {
                            return this.kubeHelper.createServiceFrom(yaml, this.getNamespace());
                        }
                    })));
                    task.title = `${task.title}...done.`;
                })
            },
        ];
    }
    /**
     * Returns list of tasks which uninstall dev-workspace.
     */
    getUninstallTasks() {
        return [
            {
                title: 'Delete all deployments for DevWorkspace controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllDeployments(this.getNamespace());
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all services for DevWorkspace controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllServices(this.getNamespace());
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all ingresses for DevWorkspace controller',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllIngresses(this.getNamespace());
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all routes for DevWorkspace controller',
                enabled: (ctx) => ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.openShiftHelper.deleteAllRoutes(this.getNamespace());
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete configmaps for DevWorkspace controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kubeHelper.getConfigMap('devworkspace-controller', this.getNamespace())) {
                        yield this.kubeHelper.deleteConfigMap('devworkspace-controller', this.getNamespace());
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete ClusterRoleBinding for DevWorkspace controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kubeHelper.clusterRoleBindingExist(this.devWorkspaceRoleBinding)) {
                        yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceRoleBinding);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete cluster Roles for DevWorkspace controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kubeHelper.clusterRoleExist(this.devWorkspaceRole)) {
                        yield this.kubeHelper.deleteClusterRole(this.devWorkspaceRole);
                    }
                    if (yield this.kubeHelper.clusterRoleExist(this.devWorkspaceEditWorkspaceClusterRole)) {
                        yield this.kubeHelper.deleteClusterRole(this.devWorkspaceEditWorkspaceClusterRole);
                    }
                    if (yield this.kubeHelper.clusterRoleExist(this.devWorkspaceViewWorkspaceClusterRole)) {
                        yield this.kubeHelper.deleteClusterRole(this.devWorkspaceViewWorkspaceClusterRole);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete service account for DevWorkspace controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kubeHelper.serviceAccountExist(this.devWorkspaceServiceAccount, this.getNamespace())) {
                        yield this.kubeHelper.deleteServiceAccount(this.devWorkspaceServiceAccount, this.getNamespace());
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete DevWorkspace controller CRDs',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kubeHelper.crdExist('components.controller.devfile.io')) {
                        yield this.kubeHelper.deleteCrd('components.controller.devfile.io');
                    }
                    if (yield this.kubeHelper.crdExist('devworkspaces.workspace.devfile.io')) {
                        yield this.kubeHelper.deleteCrd('devworkspaces.workspace.devfile.io');
                    }
                    if (yield this.kubeHelper.crdExist('devworkspacetemplates.workspace.devfile.io')) {
                        yield this.kubeHelper.deleteCrd('devworkspacetemplates.workspace.devfile.io');
                    }
                    if (yield this.kubeHelper.crdExist('workspaceroutings.controller.devfile.io')) {
                        yield this.kubeHelper.deleteCrd('workspaceroutings.controller.devfile.io');
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
        ];
    }
}
exports.DevWorkspaceTasks = DevWorkspaceTasks;
//# sourceMappingURL=devfile-workspace-operator-installer.js.map