"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const Listr = require("listr");
const context_1 = require("../../api/context");
const common_flags_1 = require("../../common-flags");
const che_1 = require("../../tasks/che");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
class Start extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Start);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            const cheTasks = new che_1.CheTasks(flags);
            const apiTasks = new api_1.ApiTasks();
            // Checks if CodeReady Workspaces is already deployed
            const preInstallTasks = new Listr([
                apiTasks.testApiTasks(flags, this),
                {
                    title: '👀  Looking for an already existing CodeReady Workspaces instance',
                    task: () => new Listr(cheTasks.checkIfCheIsInstalledTasks(flags, this))
                }
            ], ctx.listrOptions);
            const logsTasks = new Listr([{
                    title: 'Following CodeReady Workspaces logs',
                    task: () => new Listr(cheTasks.serverLogsTasks(flags, true))
                }], ctx.listrOptions);
            const startCheTasks = new Listr([{
                    title: 'Starting CodeReady Workspaces',
                    task: () => new Listr(cheTasks.scaleCheUpTasks())
                }], ctx.listrOptions);
            try {
                yield preInstallTasks.run(ctx);
                if (!ctx.isCheDeployed) {
                    cli_ux_1.cli.warn('CodeReady Workspaces has not been deployed yet. Use server:deploy command to deploy a new CodeReady Workspaces instance.');
                }
                else if (ctx.isCheReady) {
                    cli_ux_1.cli.info('CodeReady Workspaces has been already started.');
                }
                else {
                    yield logsTasks.run(ctx);
                    yield startCheTasks.run(ctx);
                    this.log(util_1.getCommandSuccessMessage());
                }
            }
            catch (err) {
                this.error(util_1.getCommandErrorMessage(err));
            }
            util_1.notifyCommandCompletedSuccessfully();
            this.exit(0);
        });
    }
}
exports.default = Start;
Start.description = 'Start CodeReady Workspaces server';
Start.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    'listr-renderer': common_flags_1.listrRenderer,
    'deployment-name': common_flags_1.cheDeployment,
    [common_flags_1.K8SPODWAITTIMEOUT_KEY]: common_flags_1.k8sPodWaitTimeout,
    [common_flags_1.K8SPODREADYTIMEOUT_KEY]: common_flags_1.k8sPodReadyTimeout,
    [common_flags_1.K8SPODDOWNLOADIMAGETIMEOUT_KEY]: common_flags_1.k8sPodDownloadImageTimeout,
    [common_flags_1.K8SPODERRORRECHECKTIMEOUT_KEY]: common_flags_1.k8sPodErrorRecheckTimeout,
    [common_flags_1.LOG_DIRECTORY_KEY]: common_flags_1.logsDirectory,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
};
//# sourceMappingURL=start.js.map