"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommonPlatformTasks = void 0;
const tslib_1 = require("tslib");
const ansi = require("ansi-colors");
const fs = require("fs-extra");
const http = require("http");
const https = require("https");
const yaml = require("js-yaml");
const kube_1 = require("../../api/kube");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
var CommonPlatformTasks;
(function (CommonPlatformTasks) {
    /**
     * Checks whether cluster on which Che is being deployed accessible.
     * Requires flags.domain to be set.
     */
    function getPingClusterTask(flags) {
        return {
            title: 'Check if cluster accessible',
            skip: () => true,
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const domain = flags.domain;
                if (!domain) {
                    task.title = `${task.title}... "--domain" flag is not set. Cannot check cluster availability.'`;
                    return;
                }
                if (!(yield checkHttpServer(domain, 80)) && !(yield checkHttpsServer(domain, 443))) {
                    throw new Error(`Cannot reach cluster at "${domain}". To skip this check add "--skip-cluster-availability-check" flag.`);
                }
                task.title = `${task.title}... ok`;
            })
        };
    }
    CommonPlatformTasks.getPingClusterTask = getPingClusterTask;
    /**
     * Sends request to given server to check if it is accessible.
     */
    function checkHttpServer(host, port) {
        return new Promise(resolve => {
            http.get({ host, port }, response => {
                // It is ok to have 404, but not 5xx
                if (response.statusCode && response.statusCode / 100 < 5) {
                    resolve(true);
                }
                resolve(false);
            }).on('error', () => {
                resolve(false);
            });
        });
    }
    function checkHttpsServer(host, port) {
        return new Promise(resolve => {
            https.get({ host, port }, response => {
                // It is ok to have 404, but not 5xx
                if (response.statusCode && response.statusCode / 100 < 5) {
                    resolve(true);
                }
                resolve(false);
            }).on('error', (err) => {
                if (err.code === 'UNABLE_TO_VERIFY_LEAF_SIGNATURE') {
                    // Connection is estabilished but the server has self-signed certificate, it's ok.
                    resolve(true);
                }
                resolve(false);
            });
        });
    }
    function oAuthProvidersExists(flags) {
        let kube = new kube_1.KubeHelper(flags);
        return {
            title: 'Verify Openshift oauth.',
            enabled: () => util_1.isOpenshiftPlatformFamily(flags.platform) && isOAuthEnabled(flags),
            task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (yield kube.isOpenShift4()) {
                    const providers = yield kube.getOpenshiftAuthProviders();
                    if (!providers || providers.length === 0) {
                        ctx.highlightedMessages.push(`❗ ${ansi.yellow('[WARNING]')} OpenShift OAuth is turned off, because there is no any identity providers configured. ${constants_1.DOCS_LINK_HOW_TO_ADD_IDENTITY_PROVIDER_OS4}`);
                        ctx.CROverrides = { spec: { auth: { openShiftoAuth: false } } };
                    }
                }
                else {
                    if ((yield kube.getAmoutUsers()) === 0) {
                        ctx.highlightedMessages.push(`❗ ${ansi.yellow('[WARNING]')} OpenShift OAuth is turned off, because there are no any users added. See: "${constants_1.DOCS_LINK_HOW_TO_CREATE_USER_OS3}"`);
                        ctx.CROverrides = { spec: { auth: { openShiftoAuth: false } } };
                    }
                }
                task.title = `${task.title}...done.`;
            })
        };
    }
    CommonPlatformTasks.oAuthProvidersExists = oAuthProvidersExists;
    /**
     * Checks if Openshift oAuth enabled in Che configuration.
     * Returns true if Openshift oAuth is enabled (or omitted) and false if it is explicitly disabled.
     */
    function isOAuthEnabled(flags) {
        if (flags['che-operator-cr-patch-yaml']) {
            const cheOperatorCrPatchYamlPath = flags['che-operator-cr-patch-yaml'];
            if (fs.existsSync(cheOperatorCrPatchYamlPath)) {
                const crPatch = yaml.safeLoad(fs.readFileSync(cheOperatorCrPatchYamlPath).toString());
                if (crPatch && crPatch.spec && crPatch.spec.auth && typeof crPatch.spec.auth.openShiftoAuth === 'boolean') {
                    return crPatch.spec.auth.openShiftoAuth;
                }
            }
        }
        if (flags['che-operator-cr-yaml']) {
            const cheOperatorCrYamlPath = flags['che-operator-cr-yaml'];
            if (fs.existsSync(cheOperatorCrYamlPath)) {
                const cr = yaml.safeLoad(fs.readFileSync(cheOperatorCrYamlPath).toString());
                if (cr && cr.spec && cr.spec.auth && typeof cr.spec.auth.openShiftoAuth === 'boolean') {
                    return cr.spec.auth.openShiftoAuth;
                }
            }
        }
        return true;
    }
})(CommonPlatformTasks = exports.CommonPlatformTasks || (exports.CommonPlatformTasks = {}));
//# sourceMappingURL=common-platform-tasks.js.map