"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const notifier = require("node-notifier");
const che_1 = require("../../api/che");
const che_api_client_1 = require("../../api/che-api-client");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
class Delete extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Delete);
            const { args } = this.parse(Delete);
            const workspaceId = args.workspace;
            let cheApiEndpoint = flags[common_flags_1.CHE_API_ENDPOINT_KEY];
            if (!cheApiEndpoint) {
                const kube = new kube_1.KubeHelper(flags);
                if (!(yield kube.hasReadPermissionsForNamespace(flags.chenamespace))) {
                    throw new Error(`CodeReady Workspaces API endpoint is required. Use flag --${common_flags_1.CHE_API_ENDPOINT_KEY} to provide it.`);
                }
                const cheHelper = new che_1.CheHelper(flags);
                cheApiEndpoint = (yield cheHelper.cheURL(flags.chenamespace)) + '/api';
            }
            const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            yield cheApiClient.checkCheApiEndpointUrl();
            const workspace = yield cheApiClient.getWorkspaceById(workspaceId, flags[common_flags_1.ACCESS_TOKEN_KEY]);
            const infrastructureNamespace = workspace.attributes.infrastructureNamespace;
            yield cheApiClient.deleteWorkspaceById(workspaceId, flags[common_flags_1.ACCESS_TOKEN_KEY]);
            cli_ux_1.cli.log(`Workspace with id '${workspaceId}' deleted.`);
            if (flags['delete-namespace']) {
                if (infrastructureNamespace === flags.chenamespace) {
                    cli_ux_1.cli.warn(`It is not possible to delete namespace '${infrastructureNamespace}' since it is used for CodeReady Workspaces deployment.`);
                    return;
                }
                const kube = new kube_1.KubeHelper(flags);
                if (yield kube.namespaceExist(infrastructureNamespace)) {
                    try {
                        yield kube.deleteNamespace(infrastructureNamespace);
                        cli_ux_1.cli.log(`Namespace '${infrastructureNamespace}' deleted.`);
                    }
                    catch (error) {
                        cli_ux_1.cli.warn(`Failed to delete namespace '${infrastructureNamespace}'. Reason: ${error.message}`);
                    }
                }
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command workspace:delete has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Delete;
Delete.description = 'delete a stopped workspace - use workspace:stop to stop the workspace before deleting it';
Delete.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    'delete-namespace': command_1.flags.boolean({
        description: 'Indicates that a Kubernetes namespace where workspace was created will be deleted as well',
        default: false
    }),
    [common_flags_1.CHE_API_ENDPOINT_KEY]: common_flags_1.cheApiEndpoint,
    [common_flags_1.ACCESS_TOKEN_KEY]: common_flags_1.accessToken,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
Delete.args = [
    {
        name: 'workspace',
        description: 'The workspace id to delete',
        required: true
    }
];
//# sourceMappingURL=delete.js.map