"use strict";
/*********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const notifier = require("node-notifier");
const che_1 = require("../../api/che");
const che_api_client_1 = require("../../api/che-api-client");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
class Start extends command_1.default {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Start);
            const { args } = this.parse(Start);
            const workspaceId = args.workspace;
            const cheHelper = new che_1.CheHelper(flags);
            let cheApiEndpoint = flags[common_flags_1.CHE_API_ENDPOINT_KEY];
            if (!cheApiEndpoint) {
                const kube = new kube_1.KubeHelper(flags);
                if (!(yield kube.hasReadPermissionsForNamespace(flags.chenamespace))) {
                    throw new Error(`CodeReady Workspaces API endpoint is required. Use flag --${common_flags_1.CHE_API_ENDPOINT_KEY} to provide it.`);
                }
                cheApiEndpoint = (yield cheHelper.cheURL(flags.chenamespace)) + '/api';
            }
            const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            yield cheApiClient.checkCheApiEndpointUrl();
            yield cheApiClient.startWorkspace(workspaceId, flags.debug, flags[common_flags_1.ACCESS_TOKEN_KEY]);
            const workspace = yield cheApiClient.getWorkspaceById(workspaceId, flags[common_flags_1.ACCESS_TOKEN_KEY]);
            if (workspace.links && workspace.links.ide) {
                const workspaceIdeURL = yield cheHelper.buildDashboardURL(workspace.links.ide);
                cli_ux_1.cli.log('Workspace start request has been sent, workspace will be available shortly:');
                cli_ux_1.cli.url(workspaceIdeURL, workspaceIdeURL);
            }
            else {
                cli_ux_1.cli.log('Workspace start request has been sent, workspace will be available shortly.');
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command workspace:start has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Start;
Start.description = 'Starts a workspace';
Start.flags = {
    help: command_1.flags.help({ char: 'h' }),
    debug: command_1.flags.boolean({
        char: 'd',
        description: 'Debug workspace start. It is useful when workspace start fails and it is needed to print more logs on startup.',
        default: false
    }),
    [common_flags_1.CHE_API_ENDPOINT_KEY]: common_flags_1.cheApiEndpoint,
    [common_flags_1.ACCESS_TOKEN_KEY]: common_flags_1.accessToken,
    chenamespace: common_flags_1.cheNamespace,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
Start.args = [
    {
        name: 'workspace',
        description: 'The workspace id to start',
        required: true
    }
];
//# sourceMappingURL=start.js.map