"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const fs = require("fs-extra");
const yaml = require("js-yaml");
const Listr = require("listr");
const notifier = require("node-notifier");
const os = require("os");
const path = require("path");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const che_1 = require("../../tasks/che");
const devfile_workspace_operator_installer_1 = require("../../tasks/component-installers/devfile-workspace-operator-installer");
const common_tasks_1 = require("../../tasks/installers/common-tasks");
const installer_1 = require("../../tasks/installers/installer");
const api_1 = require("../../tasks/platforms/api");
const common_platform_tasks_1 = require("../../tasks/platforms/common-platform-tasks");
const platform_1 = require("../../tasks/platforms/platform");
const util_1 = require("../../util");
class Start extends command_1.Command {
    setPlaformDefaults(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            flags.tls = yield this.checkTlsMode(flags);
            if (!flags.installer) {
                yield this.setDefaultInstaller(flags);
                cli_ux_1.cli.info(`› Installer type is set to: '${flags.installer}'`);
            }
            if (!flags.templates) {
                // use local templates folder if present
                const templates = 'templates';
                const templatesDir = path.resolve(templates);
                if (flags.installer === 'operator') {
                    if (fs.pathExistsSync(`${templatesDir}/codeready-operator`)) {
                        flags.templates = templatesDir;
                    }
                }
                else if (flags.installer === 'minishift-addon') {
                    if (fs.pathExistsSync(`${templatesDir}/minishift-addon/`)) {
                        flags.templates = templatesDir;
                    }
                }
                if (!flags.templates) {
                    flags.templates = path.join(__dirname, '../../../templates');
                }
            }
        });
    }
    /**
     * Determine if a directory is empty.
     */
    isDirEmpty(dirname) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return fs.readdirSync(dirname).length === 0;
                // Fails in case if directory doesn't exist
            }
            catch (_a) {
                return true;
            }
        });
    }
    /**
     * Checks if TLS is disabled via operator custom resource.
     * Returns true if TLS is enabled (or omitted) and false if it is explicitly disabled.
     */
    checkTlsMode(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (flags['che-operator-cr-patch-yaml']) {
                const cheOperatorCrPatchYamlPath = flags['che-operator-cr-patch-yaml'];
                if (fs.existsSync(cheOperatorCrPatchYamlPath)) {
                    const crPatch = yaml.safeLoad(fs.readFileSync(cheOperatorCrPatchYamlPath).toString());
                    if (crPatch && crPatch.spec && crPatch.spec.server && crPatch.spec.server.tlsSupport === false) {
                        return false;
                    }
                }
            }
            if (flags['che-operator-cr-yaml']) {
                const cheOperatorCrYamlPath = flags['che-operator-cr-yaml'];
                if (fs.existsSync(cheOperatorCrYamlPath)) {
                    const cr = yaml.safeLoad(fs.readFileSync(cheOperatorCrYamlPath).toString());
                    if (cr && cr.spec && cr.spec.server && cr.spec.server.tlsSupport === false) {
                        return false;
                    }
                }
            }
            return true;
        });
    }
    checkPlatformCompatibility(flags) {
        if (flags.installer === 'operator' && flags['che-operator-cr-yaml']) {
            const ignoredFlags = [];
            flags['plugin-registry-url'] && ignoredFlags.push('--plugin-registry-urlomain');
            flags['devfile-registry-url'] && ignoredFlags.push('--devfile-registry-url');
            flags['postgres-pvc-storage-class-name'] && ignoredFlags.push('--postgres-pvc-storage-class-name');
            flags['workspace-pvc-storage-class-name'] && ignoredFlags.push('--workspace-pvc-storage-class-name');
            flags.tls && ignoredFlags.push('--tls');
            flags.cheimage && ignoredFlags.push('--cheimage');
            flags.debug && ignoredFlags.push('--debug');
            flags.domain && ignoredFlags.push('--domain');
            flags.multiuser && ignoredFlags.push('--multiuser');
            if (ignoredFlags.length) {
                this.warn(`--che-operator-cr-yaml is used. The following flag(s) will be ignored: ${ignoredFlags.join('\t')}`);
            }
        }
        if (flags.domain && !flags['che-operator-cr-yaml'] && util_1.isOpenshiftPlatformFamily(flags.platform)) {
            this.warn('"--domain" flag is ignored for Openshift family infrastructures. It should be done on the cluster level.');
        }
        if (flags.installer) {
            if (flags.installer === 'minishift-addon') {
                if (flags.platform !== 'minishift') {
                    this.error(`🛑 Current platform is ${flags.platform}. Minishift-addon is only available for Minishift.`);
                }
            }
            else if (flags.installer === 'helm') {
                if (flags.platform !== 'k8s' && flags.platform !== 'minikube' && flags.platform !== 'microk8s' && flags.platform !== 'docker-desktop') {
                    this.error(`🛑 Current platform is ${flags.platform}. Helm installer is only available on top of Kubernetes flavor platform (including Minikube, Docker Desktop).`);
                }
            }
            if (flags.installer === 'olm' && flags.platform === 'minishift') {
                this.error(`🛑 The specified installer ${flags.installer} does not support Minishift`);
            }
            if (flags.installer !== 'olm' && flags['auto-update']) {
                this.error('"auto-update" flag should be used only with "olm" installer.');
            }
            if (flags.installer !== 'olm' && flags['starting-csv']) {
                this.error('"starting-csv" flag should be used only with "olm" installer.');
            }
            if (flags.installer !== 'olm' && flags['catalog-source-yaml']) {
                this.error('"catalog-source-yaml" flag should be used only with "olm" installer.');
            }
            if (flags.installer !== 'olm' && flags['olm-channel']) {
                this.error('"olm-channel" flag should be used only with "olm" installer.');
            }
            if (flags.installer !== 'olm' && flags['package-manifest-name']) {
                this.error('"package-manifest-name" flag should be used only with "olm" installer.');
            }
            if (flags.installer !== 'olm' && flags['catalog-source-name']) {
                this.error('"catalog-source-name" flag should be used only with "olm" installer.');
            }
            if (flags.installer !== 'olm' && flags['catalog-source-namespace']) {
                this.error('"package-manifest-name" flag should be used only with "olm" installer.');
            }
            if (flags['catalog-source-name'] && flags['catalog-source-yaml']) {
                this.error('should be provided only one argument: "catalog-source-name" or "catalog-source-yaml"');
            }
            if (!flags['package-manifest-name'] && flags['catalog-source-yaml']) {
                this.error('you need define "package-manifest-name" flag to use "catalog-source-yaml".');
            }
            if (!flags['olm-channel'] && flags['catalog-source-yaml']) {
                this.error('you need define "olm-channel" flag to use "catalog-source-yaml".');
            }
        }
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Start);
            const ctx = util_1.initializeContext();
            ctx.directory = path.resolve(flags.directory ? flags.directory : path.resolve(os.tmpdir(), 'crwctl-logs', Date.now().toString()));
            const listrOptions = { renderer: flags['listr-renderer'], collapse: false, showSubtasks: true };
            ctx.listrOptions = listrOptions;
            if (flags['self-signed-cert']) {
                this.warn('"self-signed-cert" flag is deprecated and has no effect. Autodetection is used instead.');
            }
            const cheTasks = new che_1.CheTasks(flags);
            const platformTasks = new platform_1.PlatformTasks();
            const installerTasks = new installer_1.InstallerTasks();
            const apiTasks = new api_1.ApiTasks();
            const devWorkspaceTasks = new devfile_workspace_operator_installer_1.DevWorkspaceTasks(flags);
            // Platform Checks
            let platformCheckTasks = new Listr(platformTasks.preflightCheckTasks(flags, this), listrOptions);
            platformCheckTasks.add(common_platform_tasks_1.CommonPlatformTasks.oAuthProvidersExists(flags));
            // Checks if CodeReady Workspaces is already deployed
            let preInstallTasks = new Listr(undefined, listrOptions);
            preInstallTasks.add(apiTasks.testApiTasks(flags, this));
            preInstallTasks.add({
                title: '👀  Looking for an already existing CodeReady Workspaces instance',
                task: () => new Listr(cheTasks.checkIfCheIsInstalledTasks(flags, this))
            });
            yield this.setPlaformDefaults(flags);
            let installTasks = new Listr(installerTasks.installTasks(flags, this), listrOptions);
            const startDeployedCheTasks = new Listr([{
                    title: '👀  Starting already deployed CodeReady Workspaces',
                    task: () => new Listr(cheTasks.scaleCheUpTasks(this))
                }], listrOptions);
            // Post Install Checks
            const postInstallTasks = new Listr([
                {
                    title: '✅  Post installation checklist',
                    task: () => new Listr(cheTasks.waitDeployedChe(flags, this))
                },
                {
                    title: '🧪  DevWorkspace engine (experimental / technology preview) 🚨',
                    enabled: () => flags['workspace-engine'] === 'dev-workspace',
                    task: () => new Listr(devWorkspaceTasks.getInstallTasks(flags, this))
                },
                common_tasks_1.getRetrieveKeycloakCredentialsTask(flags),
                common_tasks_1.retrieveCheCaCertificateTask(flags),
                ...cheTasks.preparePostInstallationOutput(flags),
                common_tasks_1.getPrintHighlightedMessagesTask(),
            ], listrOptions);
            const logsTasks = new Listr([{
                    title: 'Start following logs',
                    task: () => new Listr(cheTasks.serverLogsTasks(flags, true))
                }], listrOptions);
            const eventTasks = new Listr([{
                    title: 'Start following events',
                    task: () => new Listr(cheTasks.namespaceEventsTask(flags.chenamespace, this, true))
                }], listrOptions);
            try {
                yield preInstallTasks.run(ctx);
                if (!ctx.isCheDeployed) {
                    this.checkPlatformCompatibility(flags);
                    yield platformCheckTasks.run(ctx);
                    yield logsTasks.run(ctx);
                    yield eventTasks.run(ctx);
                    yield installTasks.run(ctx);
                }
                else if (!ctx.isCheReady
                    || (ctx.isPostgresDeployed && !ctx.isPostgresReady)
                    || (ctx.isKeycloakDeployed && !ctx.isKeycloakReady)
                    || (ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryReady)
                    || (ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryReady)) {
                    if (flags.platform || flags.installer) {
                        this.warn('Deployed CodeReady Workspaces is found and the specified installation parameters will be ignored');
                    }
                    // perform CodeReady Workspaces start task if there is any component that is not ready
                    yield startDeployedCheTasks.run(ctx);
                }
                yield postInstallTasks.run(ctx);
                this.log(util_1.getCommandSuccessMessage(this, ctx));
            }
            catch (err) {
                const isDirEmpty = yield this.isDirEmpty(ctx.directory);
                if (isDirEmpty) {
                    this.error(`${err}\nInstallation failed. There are no available logs.`);
                }
                this.error(`${err}\nInstallation failed, check logs in '${ctx.directory}'`);
            }
            notifier.notify({
                title: 'crwctl',
                message: util_1.getCommandSuccessMessage(this, ctx)
            });
            this.exit(0);
        });
    }
    /**
     * Sets default installer which is `olm` for OpenShift 4 with stable version of crwctl
     * and `operator` for other cases.
     */
    setDefaultInstaller(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            const isOlmPreinstalled = yield kubeHelper.isPreInstalledOLM();
            if ((flags['catalog-source-name'] || flags['catalog-source-yaml']) && isOlmPreinstalled) {
                flags.installer = 'olm';
                return;
            }
            if (flags.platform === 'openshift' && (yield kubeHelper.isOpenShift4()) && isOlmPreinstalled) {
                flags.installer = 'olm';
            }
            else {
                flags.installer = 'operator';
            }
        });
    }
}
exports.default = Start;
Start.description = 'start CodeReady Workspaces server';
Start.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    'listr-renderer': common_flags_1.listrRenderer,
    'deployment-name': common_flags_1.cheDeployment,
    cheimage: flags_1.string({
        char: 'i',
        description: 'CodeReady Workspaces server container image',
        default: constants_1.DEFAULT_CHE_IMAGE,
        env: 'CHE_CONTAINER_IMAGE'
    }),
    templates: flags_1.string({
        char: 't',
        description: 'Path to the templates folder',
        env: 'CHE_TEMPLATES_FOLDER'
    }),
    'devfile-registry-url': flags_1.string({
        description: 'The URL of the external Devfile registry.',
        env: 'CHE_WORKSPACE_DEVFILE__REGISTRY__URL'
    }),
    'plugin-registry-url': flags_1.string({
        description: 'The URL of the external plugin registry.',
        env: 'CHE_WORKSPACE_PLUGIN__REGISTRY__URL'
    }),
    cheboottimeout: flags_1.string({
        char: 'o',
        description: 'CodeReady Workspaces server bootstrap timeout (in milliseconds)',
        default: '40000',
        required: true,
        env: 'CHE_SERVER_BOOT_TIMEOUT'
    }),
    k8spodwaittimeout: flags_1.string({
        description: 'Waiting time for Pod Wait Timeout Kubernetes (in milliseconds)',
        default: '300000'
    }),
    k8spodreadytimeout: flags_1.string({
        description: 'Waiting time for Pod Ready Kubernetes (in milliseconds)',
        default: '130000'
    }),
    multiuser: command_1.flags.boolean({
        char: 'm',
        description: `Starts CodeReady Workspaces in multi-user mode.
 		                Note, this option is turned on by default.`,
        default: false
    }),
    tls: command_1.flags.boolean({
        char: 's',
        description: `Enable TLS encryption.
                    Note, this option is turned on by default.
                    To provide own certificate for Kubernetes infrastructure, 'che-tls' secret with TLS certificate must be pre-created in the configured namespace.
                    In case of providing own self-signed certificate 'self-signed-certificate' secret should be also created.
                    For OpenShift, router will use default cluster certificates.
                    Please see the docs how to deploy CodeReady Workspaces on different infrastructures: ${constants_1.DOCS_LINK_INSTALL_RUNNING_CHE_LOCALLY}`
    }),
    'self-signed-cert': command_1.flags.boolean({
        description: 'Deprecated. The flag is ignored. Self signed certificates usage is autodetected now.',
        default: false
    }),
    platform: flags_1.string({
        char: 'p',
        description: 'Type of OpenShift platform. Valid values are \"openshift\", \"crc (for CodeReady Containers)\".',
        options: ['openshift', 'crc'],
        default: 'openshift'
    }),
    installer: flags_1.string({
        char: 'a',
        description: 'Installer type. If not set, default is olm for OpenShift >= 4.2, and operator for earlier versions.',
        options: ['olm', 'operator']
    }),
    debug: flags_1.boolean({
        description: 'Enables the debug mode for CodeReady Workspaces server. To debug CodeReady Workspaces server from localhost use \'server:debug\' command.',
        default: false
    }),
    'che-operator-image': flags_1.string({
        description: 'Container image of the operator. This parameter is used only when the installer is the operator',
        default: constants_1.DEFAULT_CHE_OPERATOR_IMAGE
    }),
    'che-operator-cr-yaml': flags_1.string({
        description: 'Path to a yaml file that defines a CheCluster used by the operator. This parameter is used only when the installer is the \'operator\' or the \'olm\'.',
        default: ''
    }),
    'che-operator-cr-patch-yaml': flags_1.string({
        description: 'Path to a yaml file that overrides the default values in CheCluster CR used by the operator. This parameter is used only when the installer is the \'operator\' or the \'olm\'.',
        default: ''
    }),
    directory: flags_1.string({
        char: 'd',
        description: 'Directory to store logs into',
        env: 'CHE_LOGS'
    }),
    'workspace-pvc-storage-class-name': flags_1.string({
        description: 'persistent volume(s) storage class name to use to store CodeReady Workspaces workspaces data',
        env: 'CHE_INFRA_KUBERNETES_PVC_STORAGE__CLASS__NAME',
        default: ''
    }),
    'postgres-pvc-storage-class-name': flags_1.string({
        description: 'persistent volume storage class name to use to store CodeReady Workspaces postgres database',
        default: ''
    }),
    'skip-version-check': command_1.flags.boolean({
        description: 'Skip minimal versions check.',
        default: false
    }),
    'skip-cluster-availability-check': command_1.flags.boolean({
        description: 'Skip cluster availability check. The check is a simple request to ensure the cluster is reachable.',
        default: false
    }),
    'auto-update': command_1.flags.boolean({
        description: `Auto update approval strategy for installation CodeReady Workspaces.
                    With this strategy will be provided auto-update CodeReady Workspaces without any human interaction.
                    By default strategy this flag is false. It requires approval from user.
                    To approve installation newer version CodeReady Workspaces user should execute 'crwctl server:update' command.
                    This parameter is used only when the installer is 'olm'.`,
        default: false,
        exclusive: ['starting-csv']
    }),
    'starting-csv': command_1.flags.string({
        description: `Starting cluster service version(CSV) for installation CodeReady Workspaces.
                    Flags uses to set up start installation version Che.
                    For example: 'starting-csv' provided with value 'eclipse-che.v7.10.0' for stable channel.
                    Then OLM will install CodeReady Workspaces with version 7.10.0.
                    Notice: this flag will be ignored with 'auto-update' flag. OLM with auto-update mode installs the latest known version.
                    This parameter is used only when the installer is 'olm'.`
    }),
    'olm-channel': flags_1.string({
        description: `Olm channel to install CodeReady Workspaces, f.e. stable.
                    If options was not set, will be used default version for package manifest.
                    This parameter is used only when the installer is the 'olm'.`,
    }),
    'package-manifest-name': flags_1.string({
        description: `Package manifest name to subscribe to CodeReady Workspaces OLM package manifest.
                    This parameter is used only when the installer is the 'olm'.`,
    }),
    'catalog-source-yaml': flags_1.string({
        description: `Path to a yaml file that describes custom catalog source for installation CodeReady Workspaces operator.
                    Catalog source will be applied to the namespace with Che operator.
                    Also you need define 'olm-channel' name and 'package-manifest-name'.
                    This parameter is used only when the installer is the 'olm'.`,
    }),
    'catalog-source-name': flags_1.string({
        description: `OLM catalog source to install CodeReady Workspaces operator.
                    This parameter is used only when the installer is the 'olm'.`
    }),
    'catalog-source-namespace': flags_1.string({
        description: `Namespace for OLM catalog source to install CodeReady Workspaces operator.
                    This parameter is used only when the installer is the 'olm'.`
    }),
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    'workspace-engine': flags_1.string({
        description: 'Workspace Engine. If not set, default is "che-server". "dev-workspace" is experimental.',
        options: ['che-server', 'dev-workspace'],
        default: 'che-server',
    }),
    'dev-workspace-controller-image': flags_1.string({
        description: 'Container image of the dev workspace controller. This parameter is used only when the workspace engine is the DevWorkspace',
        default: constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_IMAGE,
        env: 'DEV_WORKSPACE_OPERATOR_IMAGE',
    }),
    'dev-workspace-controller-namespace': common_flags_1.devWorkspaceControllerNamespace
};
//# sourceMappingURL=start.js.map