"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.OLMTasks = void 0;
const tslib_1 = require("tslib");
const cli_ux_1 = require("cli-ux");
const Listr = require("listr");
const kube_1 = require("../../api/kube");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const common_tasks_1 = require("./common-tasks");
class OLMTasks {
    /**
     * Returns list of tasks which perform preflight platform checks.
     */
    startTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            this.isOlmPreInstalledTask(command, kube),
            common_tasks_1.copyOperatorResources(flags, command.config.cacheDir),
            common_tasks_1.createNamespaceTask(flags.chenamespace, flags.platform),
            {
                title: 'Create operator group',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.operatorGroupExists(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace)) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createOperatorGroup(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Configure context information',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.defaultCatalogSourceNamespace = util_1.isKubernetesPlatformFamily(flags.platform) ? constants_1.DEFAULT_OLM_KUBERNETES_NAMESPACE : constants_1.DEFAULT_OPENSHIFT_MARKET_PLACE_NAMESPACE;
                    // catalog source name for stable Che version
                    ctx.catalogSourceNameStable = util_1.isKubernetesPlatformFamily(flags.platform) ? constants_1.KUBERNETES_OLM_CATALOG : constants_1.OPENSHIFT_OLM_CATALOG;
                    if (!flags['auto-update'] && !util_1.isStableVersion(flags)) {
                        ctx.approvalStarategy = 'Automatic';
                    }
                    else {
                        ctx.approvalStarategy = flags['auto-update'] ? 'Automatic' : 'Manual';
                    }
                    ctx.sourceName = flags['catalog-source-name'] || constants_1.CUSTOM_CATALOG_SOURCE_NAME;
                    ctx.generalPlatformName = util_1.isKubernetesPlatformFamily(flags.platform) ? 'kubernetes' : 'openshift';
                    task.title = `${task.title}...done.`;
                })
            },
            {
                enabled: () => !util_1.isStableVersion(flags) && !flags['catalog-source-name'] && !flags['catalog-source-yaml'],
                title: `Create nightly index CatalogSource in the namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.catalogSourceExists(constants_1.NIGHTLY_CATALOG_SOURCE_NAME, flags.chenamespace))) {
                        const catalogSourceImage = `quay.io/eclipse/eclipse-che-${ctx.generalPlatformName}-opm-catalog:preview`;
                        const nigthlyCatalogSource = this.constructIndexCatalogSource(flags.chenamespace, catalogSourceImage);
                        yield kube.createCatalogSource(nigthlyCatalogSource);
                        yield kube.waitCatalogSource(flags.chenamespace, constants_1.NIGHTLY_CATALOG_SOURCE_NAME);
                    }
                    else {
                        task.title = `${task.title}...It already exists.`;
                    }
                })
            },
            {
                enabled: () => flags['catalog-source-yaml'],
                title: 'Create custom catalog source from file',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const customCatalogSource = kube.readCatalogSourceFromFile(flags['catalog-source-yaml']);
                    if (!(yield kube.catalogSourceExists(customCatalogSource.metadata.name, flags.chenamespace))) {
                        customCatalogSource.metadata.name = ctx.sourceName;
                        customCatalogSource.metadata.namespace = flags.chenamespace;
                        yield kube.createCatalogSource(customCatalogSource);
                        yield kube.waitCatalogSource(flags.chenamespace, constants_1.CUSTOM_CATALOG_SOURCE_NAME);
                        task.title = `${task.title}...created new one, with name ${constants_1.CUSTOM_CATALOG_SOURCE_NAME} in the namespace ${flags.chenamespace}.`;
                    }
                    else {
                        task.title = `${task.title}...It already exists.`;
                    }
                })
            },
            {
                title: 'Create operator subscription',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.operatorSubscriptionExists(constants_1.SUBSCRIPTION_NAME, flags.chenamespace)) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        let subscription;
                        // stable Che CatalogSource
                        if (util_1.isStableVersion(flags)) {
                            subscription = this.constructSubscription(constants_1.SUBSCRIPTION_NAME, constants_1.DEFAULT_CHE_OLM_PACKAGE_NAME, flags.chenamespace, ctx.defaultCatalogSourceNamespace, constants_1.OLM_STABLE_CHANNEL_NAME, ctx.catalogSourceNameStable, ctx.approvalStarategy, flags['starting-csv']);
                            // custom Che CatalogSource
                        }
                        else if (flags['catalog-source-yaml'] || flags['catalog-source-name']) {
                            const catalogSourceNamespace = flags['catalog-source-namespace'] || flags.chenamespace;
                            subscription = this.constructSubscription(constants_1.SUBSCRIPTION_NAME, flags['package-manifest-name'], flags.chenamespace, catalogSourceNamespace, flags['olm-channel'], ctx.sourceName, ctx.approvalStarategy, flags['starting-csv']);
                            // nightly Che CatalogSource
                        }
                        else {
                            subscription = this.constructSubscription(constants_1.SUBSCRIPTION_NAME, `eclipse-che-preview-${ctx.generalPlatformName}`, flags.chenamespace, flags.chenamespace, constants_1.OLM_NIGHTLY_CHANNEL_NAME, constants_1.NIGHTLY_CATALOG_SOURCE_NAME, ctx.approvalStarategy, flags['starting-csv']);
                        }
                        yield kube.createOperatorSubscription(subscription);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Wait while subscription is ready',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const installPlan = yield kube.waitOperatorSubscriptionReadyForApproval(flags.chenamespace, constants_1.SUBSCRIPTION_NAME, 600);
                    ctx.installPlanName = installPlan.name;
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Approve installation',
                enabled: ctx => ctx.approvalStarategy === 'Manual',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.approveOperatorInstallationPlan(ctx.installPlanName, flags.chenamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait while operator installed',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.waitUntilOperatorIsInstalled(ctx.installPlanName, flags.chenamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            common_tasks_1.createEclipseCheCluster(flags, kube)
        ], { renderer: flags['listr-renderer'] });
    }
    preUpdateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            this.isOlmPreInstalledTask(command, kube),
            {
                title: 'Check if operator group exists',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.operatorGroupExists(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace))) {
                        command.error(`Unable to find operator group ${constants_1.OPERATOR_GROUP_NAME}`);
                    }
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Check if operator subscription exists',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.operatorSubscriptionExists(constants_1.SUBSCRIPTION_NAME, flags.chenamespace))) {
                        command.error(`Unable to find operator subscription ${constants_1.SUBSCRIPTION_NAME}`);
                    }
                    task.title = `${task.title}...done.`;
                })
            },
        ], { renderer: flags['listr-renderer'] });
    }
    updateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Get operator installation plan',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const subscription = yield kube.getOperatorSubscription(constants_1.SUBSCRIPTION_NAME, flags.chenamespace);
                    if (subscription.status) {
                        if (subscription.status.state === 'AtLatestKnown') {
                            task.title = `Everything is up to date. Installed the latest known version '${subscription.status.currentCSV}'.`;
                            return;
                        }
                        if (subscription.status.state === 'UpgradePending' && subscription.status.conditions) {
                            const installCondition = subscription.status.conditions.find(condition => condition.type === 'InstallPlanPending' && condition.status === 'True');
                            if (installCondition) {
                                ctx.installPlanName = subscription.status.installplan.name;
                                task.title = `${task.title}...done.`;
                                return;
                            }
                        }
                    }
                    command.error('Unable to find installation plan to update.');
                })
            },
            {
                title: 'Approve installation',
                enabled: (ctx) => ctx.installPlanName,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.approveOperatorInstallationPlan(ctx.installPlanName, flags.chenamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait while newer operator installed',
                enabled: (ctx) => ctx.installPlanName,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.waitUntilOperatorIsInstalled(ctx.installPlanName, flags.chenamespace, 60);
                    task.title = `${task.title}...done.`;
                })
            },
        ], { renderer: flags['listr-renderer'] });
    }
    deleteTasks(flags) {
        const kube = new kube_1.KubeHelper(flags);
        return [
            {
                title: 'Check if OLM is pre-installed on the platform',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.isPreInstalledOLM = (yield kube.isPreInstalledOLM()) ? true : false;
                    task.title = `${task.title}: ${ctx.isPreInstalledOLM}...OK`;
                })
            },
            {
                title: `Delete(OLM) operator subscription ${constants_1.SUBSCRIPTION_NAME}`,
                enabled: ctx => ctx.isPreInstalledOLM,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.operatorSubscriptionExists(constants_1.SUBSCRIPTION_NAME, flags.chenamespace)) {
                        yield kube.deleteOperatorSubscription(constants_1.SUBSCRIPTION_NAME, flags.chenamespace);
                    }
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: 'Delete(OLM) CodeReady Workspaces cluster service versions',
                enabled: ctx => ctx.isPreInstalledOLM,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const csvs = yield kube.getClusterServiceVersions(flags.chenamespace);
                    const csvsToDelete = csvs.items.filter(csv => csv.metadata.name.startsWith(constants_1.CVS_PREFIX));
                    csvsToDelete.forEach(csv => kube.deleteClusterServiceVersion(flags.chenamespace, csv.metadata.name));
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete(OLM) operator group ${constants_1.OPERATOR_GROUP_NAME}`,
                enabled: ctx => ctx.isPreInstalledOLM,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.operatorGroupExists(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace)) {
                        yield kube.deleteOperatorGroup(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace);
                    }
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete(OLM) custom catalog source ${constants_1.CUSTOM_CATALOG_SOURCE_NAME}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.catalogSourceExists(constants_1.CUSTOM_CATALOG_SOURCE_NAME, flags.chenamespace)) {
                        yield kube.deleteCatalogSource(flags.chenamespace, constants_1.CUSTOM_CATALOG_SOURCE_NAME);
                    }
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete(OLM) nigthly catalog source ${constants_1.NIGHTLY_CATALOG_SOURCE_NAME}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.catalogSourceExists(constants_1.NIGHTLY_CATALOG_SOURCE_NAME, flags.chenamespace)) {
                        yield kube.deleteCatalogSource(flags.chenamespace, constants_1.NIGHTLY_CATALOG_SOURCE_NAME);
                    }
                    task.title = `${task.title}...OK`;
                })
            }
        ];
    }
    isOlmPreInstalledTask(command, kube) {
        return {
            title: 'Check if OLM is pre-installed on the platform',
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (!(yield kube.isPreInstalledOLM())) {
                    cli_ux_1.cli.warn('Looks like your platform hasn\'t got embedded OLM, so you should install it manually. For quick start you can use:');
                    cli_ux_1.cli.url('install.sh', 'https://raw.githubusercontent.com/operator-framework/operator-lifecycle-manager/master/deploy/upstream/quickstart/install.sh');
                    command.error('OLM is required for installation CodeReady Workspaces with installer flag \'olm\'');
                }
                task.title = `${task.title}...done.`;
            })
        };
    }
    constructSubscription(name, packageName, namespace, sourceNamespace, channel, sourceName, installPlanApproval, startingCSV) {
        return {
            apiVersion: 'operators.coreos.com/v1alpha1',
            kind: 'Subscription',
            metadata: {
                name,
                namespace
            },
            spec: {
                channel,
                installPlanApproval,
                name: packageName,
                source: sourceName,
                sourceNamespace,
                startingCSV,
            }
        };
    }
    constructIndexCatalogSource(namespace, catalogSourceImage) {
        return {
            apiVersion: 'operators.coreos.com/v1alpha1',
            kind: 'CatalogSource',
            metadata: {
                name: constants_1.NIGHTLY_CATALOG_SOURCE_NAME,
                namespace,
            },
            spec: {
                image: catalogSourceImage,
                sourceType: 'grpc',
                updateStrategy: {
                    registryPoll: {
                        interval: '15m'
                    }
                }
            }
        };
    }
}
exports.OLMTasks = OLMTasks;
//# sourceMappingURL=olm.js.map