"use strict";
/**
 * Copyright (c) 2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseBackupServerConfig = exports.requestRestore = exports.requestBackup = exports.getBackupServerType = exports.SSH_KEY_SECRET_NAME = exports.AWS_CREDENTIALS_SECRET_NAME = exports.REST_SERVER_CREDENTIALS_SECRET_NAME = exports.BACKUP_REPOSITORY_PASSWORD_SECRET_NAME = exports.BACKUP_SERVER_CONFIG_NAME = exports.RESTORE_CR_NAME = exports.BACKUP_CR_NAME = void 0;
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const constants_1 = require("../constants");
const kube_1 = require("./kube");
exports.BACKUP_CR_NAME = 'eclipse-che-backup';
exports.RESTORE_CR_NAME = 'eclipse-che-restore';
exports.BACKUP_SERVER_CONFIG_NAME = 'eclipse-che-backup-server-config';
exports.BACKUP_REPOSITORY_PASSWORD_SECRET_NAME = 'crwctl-backup-repository-password';
exports.REST_SERVER_CREDENTIALS_SECRET_NAME = 'crwctl-backup-rest-server-credentials';
exports.AWS_CREDENTIALS_SECRET_NAME = 'crwctl-aws-credentials';
exports.SSH_KEY_SECRET_NAME = 'crwctl-backup-sftp-server-key';
/**
 * Detects backup server type. Returns empty if there is no type specified or type is invalid.
 * @param url full url to backup server including restic protocol, e.g. sftp://url
 */
function getBackupServerType(url) {
    if (url.startsWith('rest:')) {
        return 'rest';
    }
    else if (url.startsWith('s3:')) {
        return 's3';
    }
    else if (url.startsWith('sftp:')) {
        return 'sftp';
    }
    return '';
}
exports.getBackupServerType = getBackupServerType;
/**
 * Submits backup of Che installation task.
 * @param namespace namespace in which Che is installed
 * @param backupServerConfig backup server configuration data or name of the config CR
 */
function requestBackup(namespace, backupServerConfig) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const kube = new kube_1.KubeHelper();
        const backupServerConfigName = yield getBackupServerConfigurationName(namespace, backupServerConfig);
        return kube.recreateBackupCr(namespace, exports.BACKUP_CR_NAME, backupServerConfigName);
    });
}
exports.requestBackup = requestBackup;
/**
 * Submits Che restore task.
 * @param namespace namespace in which Che should be restored
 * @param backupServerConfig backup server configuration data or name of the config CR
 */
function requestRestore(namespace, backupServerConfig, snapshotId) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const kube = new kube_1.KubeHelper();
        const backupServerConfigName = yield getBackupServerConfigurationName(namespace, backupServerConfig);
        if (!backupServerConfigName) {
            throw new Error(`No backup server configuration found in ${namespace} namespace`);
        }
        return kube.recreateRestoreCr(namespace, exports.RESTORE_CR_NAME, backupServerConfigName, snapshotId);
    });
}
exports.requestRestore = requestRestore;
/**
 * Returns backup server configuration object name by backup server configuration data,
 * or checks that bacлup server configuration with given name exists.
 * This function may create a new backup server configuration or replace existing according to the given data.
 * Returns empty string if there is no data to create new one or choose from existing backup server configurations.
 * @param namespace namespace with backup server configuration
 * @param backupServerConfig backup server configuration data or name of the backup server config CR
 * @returns name of existing backup server configuration in the given namespace or empty string if none suitable
 */
function getBackupServerConfigurationName(namespace, backupServerConfig) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const kube = new kube_1.KubeHelper();
        if (backupServerConfig) {
            if (typeof backupServerConfig === 'string') {
                // Name of CR with backup server configuration provided
                // Check if it exists
                const backupServerConfigCr = yield kube.getCustomResource(namespace, constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_BACKUP_SERVER_CONFIG_KIND_PLURAL);
                if (!backupServerConfigCr) {
                    throw new Error(`Backup server configuration with '${backupServerConfig}' name not found in '${namespace}' namespace.`);
                }
                return backupServerConfig;
            }
            else {
                // Backup server configuration provided
                const backupServerConfigCrYaml = parseBackupServerConfig(backupServerConfig);
                backupServerConfigCrYaml.metadata = new client_node_1.V1ObjectMeta();
                backupServerConfigCrYaml.metadata.namespace = namespace;
                backupServerConfigCrYaml.metadata.name = exports.BACKUP_SERVER_CONFIG_NAME;
                yield provisionCredentialsSecrets(namespace, backupServerConfig);
                yield kube.recreateCheGroupCr(backupServerConfigCrYaml, constants_1.CHE_BACKUP_SERVER_CONFIG_KIND_PLURAL);
                return exports.BACKUP_SERVER_CONFIG_NAME;
            }
        }
        // No backup server configuration provided.
        // Read all existing backup server configurations within the namespace.
        const backupServerConfigs = yield kube.getAllCustomResources(constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_BACKUP_SERVER_CONFIG_KIND_PLURAL);
        switch (backupServerConfigs.length) {
            case 0:
                // There is no backup server configurations
                return '';
            case 1:
                // There is only one available backup server configuration, use it
                return backupServerConfigs[0].metadata.name;
            default:
                // There are many backup server configurations available, use one created by crwctl, if any
                const backupServerConfigCr = backupServerConfigs.find(cr => cr.metadata.name === exports.BACKUP_SERVER_CONFIG_NAME);
                if (!backupServerConfigCr) {
                    throw new Error(`Too many backup servers configurations in '${namespace}'`);
                }
                return exports.BACKUP_SERVER_CONFIG_NAME;
        }
    });
}
function parseBackupServerConfig(backupServerConfig) {
    const backupServerConfigCrYaml = {
        apiVersion: `${constants_1.CHE_CLUSTER_API_GROUP}/${constants_1.CHE_CLUSTER_API_VERSION}`,
        kind: 'CheBackupServerConfiguration',
        spec: {},
    };
    const serverType = getBackupServerType(backupServerConfig.url);
    switch (serverType) {
        case 'rest':
            // Docs: https://restic.readthedocs.io/en/latest/030_preparing_a_new_repo.html#rest-server
            // Example urls: rest://host:5000/repo/ rest://https://user:password@host:5000/repo/
            const restRegex = /rest:(?:\/\/)?(?:(?<protocol>https?):\/\/)?(?:(?<credentials>.+)\@)?(?<hostname>[\w\.\-]+)(?::(?<port>\d{1,5}))?(?:\/(?<path>[\w\-\.\/]*))?/g;
            const restUrlMatch = restRegex.exec(backupServerConfig.url);
            if (!restUrlMatch || !restUrlMatch.groups) {
                throw new Error(`Invalid REST server url: '${backupServerConfig.url}'`);
            }
            const restCfg = {
                hostname: restUrlMatch.groups.hostname,
                repositoryPath: restUrlMatch.groups.path,
                repositoryPasswordSecretRef: exports.BACKUP_REPOSITORY_PASSWORD_SECRET_NAME,
                protocol: restUrlMatch.groups.protocol,
                port: parseInt(restUrlMatch.groups.port, 10),
            };
            if (restUrlMatch.groups.credentials) {
                const credentials = restUrlMatch.groups.credentials;
                const colonIndex = credentials.indexOf(':');
                if (colonIndex === -1) {
                    throw new Error(`Invalid REST server url: '${backupServerConfig.url}'`);
                }
                const username = credentials.substring(0, colonIndex);
                const password = credentials.substring(colonIndex + 1);
                backupServerConfig.credentials = { username, password };
            }
            if (backupServerConfig.credentials) {
                restCfg.credentialsSecretRef = exports.REST_SERVER_CREDENTIALS_SECRET_NAME;
            }
            backupServerConfigCrYaml.spec.rest = restCfg;
            break;
        case 's3':
            // Docs: https://restic.readthedocs.io/en/latest/030_preparing_a_new_repo.html#amazon-s3
            // Example urls: s3://s3.amazonaws.com/bucket/repo s3://http://server:port/bucket/repo
            const awsRegex = /^s3:(?:\/\/)?((?<protocol>https?):\/\/)?(?<hostname>[\w\.\-]+)?(?::(?<port>\d{1,5}))?\/(?<path>[\w\-\/]+)$/g;
            const s3UrlMatch = awsRegex.exec(backupServerConfig.url);
            if (!s3UrlMatch || !s3UrlMatch.groups) {
                throw new Error(`Invalid S3 server url: '${backupServerConfig.url}'`);
            }
            const s3Cfg = {
                repositoryPath: s3UrlMatch.groups.path,
                repositoryPasswordSecretRef: exports.BACKUP_REPOSITORY_PASSWORD_SECRET_NAME,
                awsAccessKeySecretRef: exports.AWS_CREDENTIALS_SECRET_NAME,
                hostname: s3UrlMatch.groups.hostname,
                protocol: s3UrlMatch.groups.protocol,
                port: parseInt(s3UrlMatch.groups.port, 10),
            };
            backupServerConfigCrYaml.spec.awss3 = s3Cfg;
            break;
        case 'sftp':
            // Docs: https://restic.readthedocs.io/en/latest/030_preparing_a_new_repo.html#sftp
            // Example urls: sftp:user@host:/srv/repo sftp://user@host:1234//srv/repo
            const sftpRegex = /^sftp:(?:\/\/)?(?<username>\w+)\@(?<hostname>[\w\.\-]+):(?:(?<port>\d{1,5})\/)?(?<path>[\w\-\/]+)$/g;
            const sftpUrlMatch = sftpRegex.exec(backupServerConfig.url);
            if (!sftpUrlMatch || !sftpUrlMatch.groups) {
                throw new Error(`Invalid SFTP server url: '${backupServerConfig.url}'`);
            }
            const sftpCfg = {
                username: sftpUrlMatch.groups.username,
                hostname: sftpUrlMatch.groups.hostname,
                repositoryPath: sftpUrlMatch.groups.path,
                repositoryPasswordSecretRef: exports.BACKUP_REPOSITORY_PASSWORD_SECRET_NAME,
                sshKeySecretRef: exports.SSH_KEY_SECRET_NAME,
                port: parseInt(sftpUrlMatch.groups.port, 10),
            };
            backupServerConfigCrYaml.spec.sftp = sftpCfg;
            break;
        default:
            throw new Error(`Unknown backup server type: '${serverType}'`);
    }
    return backupServerConfigCrYaml;
}
exports.parseBackupServerConfig = parseBackupServerConfig;
/**
 * Creates or replaces secrets in the target namespace with user provided credentials.
 * @param namespace namespace in which secrets should be created
 * @param backupServerConfig credentials provided by user
 */
function provisionCredentialsSecrets(namespace, backupServerConfig) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const kube = new kube_1.KubeHelper();
        const data = { 'repo-password': backupServerConfig.repoPassword };
        yield kube.createOrReplaceSecret(namespace, exports.BACKUP_REPOSITORY_PASSWORD_SECRET_NAME, data);
        if (backupServerConfig.credentials) {
            const serverType = getBackupServerType(backupServerConfig.url);
            switch (serverType) {
                case 'rest':
                    const restServerCredentials = backupServerConfig.credentials;
                    const username = restServerCredentials.username;
                    const password = restServerCredentials.password;
                    if (username && password) {
                        const data = { username, password };
                        yield kube.createOrReplaceSecret(namespace, exports.REST_SERVER_CREDENTIALS_SECRET_NAME, data);
                    }
                    break;
                case 's3':
                    const awsCredentials = backupServerConfig.credentials;
                    const awsAccessKeyId = awsCredentials.awsAccessKeyId;
                    const awsSecretAccessKey = awsCredentials.awsSecretAccessKey;
                    if (awsAccessKeyId && awsSecretAccessKey) {
                        const data = { awsAccessKeyId, awsSecretAccessKey };
                        yield kube.createOrReplaceSecret(namespace, exports.AWS_CREDENTIALS_SECRET_NAME, data);
                    }
                    break;
                case 'sftp':
                    const sftpServerCredentials = backupServerConfig.credentials;
                    if (sftpServerCredentials.sshKey) {
                        const data = { 'ssh-privatekey': sftpServerCredentials.sshKey };
                        yield kube.createOrReplaceSecret(namespace, exports.SSH_KEY_SECRET_NAME, data);
                    }
                    break;
                default:
                    throw new Error(`Unknown backup server type: '${serverType}'`);
            }
        }
    });
}
//# sourceMappingURL=backup-restore.js.map