"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/command/lib/flags");
const cli_ux_1 = require("cli-ux");
const Listrq = require("listr");
const Listr = require("listr");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const che_1 = require("../../tasks/che");
const devfile_workspace_operator_installer_1 = require("../../tasks/component-installers/devfile-workspace-operator-installer");
const olm_1 = require("../../tasks/installers/olm");
const operator_1 = require("../../tasks/installers/operator");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
class Delete extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Delete);
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Delete.id, flags });
            if (flags['skip-deletion-check']) {
                this.warn('\'--skip-deletion-check\' flag is deprecated, use \'--yes\' instead.');
                flags.yes = flags['skip-deletion-check'];
            }
            const apiTasks = new api_1.ApiTasks();
            const kube = new kube_1.KubeHelper(flags);
            const operatorTasks = new operator_1.OperatorTasks();
            const olmTasks = new olm_1.OLMTasks();
            const cheTasks = new che_1.CheTasks(flags);
            const devWorkspaceTasks = new devfile_workspace_operator_installer_1.DevWorkspaceTasks(flags);
            const tasks = new Listrq([], ctx.listrOptions);
            tasks.add(apiTasks.testApiTasks(flags));
            tasks.add(operatorTasks.deleteTasks(flags));
            tasks.add(olmTasks.deleteTasks(flags));
            tasks.add(cheTasks.deleteTasks(flags));
            tasks.add(cheTasks.waitPodsDeletedTasks());
            // Remove devworkspace controller only if there are no more cheClusters after olm/operator tasks
            tasks.add({
                title: 'Uninstall DevWorkspace Controller and DevWorkspace Che Controller',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const checlusters = yield kube.getAllCheClusters();
                    if (checlusters.length === 0) {
                        return new Listr(devWorkspaceTasks.getUninstallTasks());
                    }
                    task.title = `${task.title}...Skipped: another CodeReady Workspaces deployment found.`;
                })
            });
            if (flags['delete-namespace']) {
                tasks.add(cheTasks.deleteNamespace(flags));
            }
            if (flags.batch || (yield this.isDeletionConfirmed(flags))) {
                try {
                    yield tasks.run();
                    cli_ux_1.cli.log(util_1.getCommandSuccessMessage());
                }
                catch (err) {
                    this.error(util_1.wrapCommandError(err));
                }
            }
            else {
                this.exit(0);
            }
            util_1.notifyCommandCompletedSuccessfully();
            this.exit(0);
        });
    }
    isDeletionConfirmed(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kc = new kube_1.KubeHelper(flags);
            const cluster = kc.kubeConfig.getCurrentCluster();
            if (!cluster) {
                throw new Error('Failed to get current Kubernetes cluster. Check if the current context is set via kubectl/oc');
            }
            if (!flags.batch && !flags.yes) {
                return cli_ux_1.cli.confirm(`You're going to remove CodeReady Workspaces server in namespace '${flags.chenamespace}' on server '${cluster ? cluster.server : ''}'. If you want to continue - press Y`);
            }
            return true;
        });
    }
}
exports.default = Delete;
Delete.description = 'delete any CodeReady Workspaces related resource';
Delete.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    batch: common_flags_1.batch,
    'delete-namespace': flags_1.boolean({
        description: 'Indicates that a CodeReady Workspaces namespace will be deleted as well',
        default: false,
    }),
    'deployment-name': common_flags_1.cheDeployment,
    'listr-renderer': common_flags_1.listrRenderer,
    'skip-deletion-check': flags_1.boolean({
        description: 'Skip user confirmation on deletion check',
        default: false,
        hidden: true,
    }),
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    yes: common_flags_1.assumeYes,
    telemetry: common_flags_1.CHE_TELEMETRY,
};
//# sourceMappingURL=delete.js.map