package com.redhat.installer.installation.action;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.PanelAction;
import com.izforge.izpack.installer.PanelActionConfiguration;
import com.izforge.izpack.util.AbstractUIHandler;
import com.redhat.installer.ports.utils.PortOffset;

import java.util.HashMap;
import java.util.Map;

/**
 * Sets variables in idata associated with shortcuts.
 * @author Dustin Kut Moy Cheung
 */
public class ShortcutHelper implements PanelAction
{

    private static final String STANDALONE_MANAGEMENT_HTTP_KEY = "standalone.xml.management-http";
    private static final String STANDALONE_MANAGEMENT_HTTPS_KEY = "standalone.xml.management-https";

    private static final String DOMAIN_MANAGEMENT_HTTP_KEY = "host.management-http";
    private static final String DOMAIN_MANAGEMENT_HTTPS_KEY = "domain.default.management-https-2";

    private static final String STANDALONE_SHORTCUT_KEY = "standalone.shortcut.port";
    private static final String DOMAIN_SHORTCUT_KEY = "domain.shortcut.port";

    private static final String URL_PROTOCOL_KEY = "url.protocol";

    private Map<String, String> portMap = new HashMap<String, String>();

    private static final String SSL_CONDITION = "install.ssl";
    private AutomatedInstallData idata;

    /**
     * Entry point to run this action
     * @param handler Note, as per spec., if there isn't a GUI, handler is null
     */
    public void executeAction(final AutomatedInstallData idata, AbstractUIHandler handler) {
        this.idata = idata;
        fillPortMap();
        applyOffset();
        applySSL();
    }

    private void fillPortMap() {
        portMap.put(STANDALONE_MANAGEMENT_HTTP_KEY, null);
        portMap.put(STANDALONE_MANAGEMENT_HTTPS_KEY, null);
        portMap.put(DOMAIN_MANAGEMENT_HTTP_KEY, null);
        portMap.put(DOMAIN_MANAGEMENT_HTTPS_KEY, null);
    }


    private void applyOffset() {
        int portOffset = PortOffset.get();

        portMap.replaceAll((k, v) -> {
            int portNumber = getNumber(getPortNumber(idata.getVariable(k)));
            portNumber += portOffset;
            return String.valueOf(portNumber);
        });
    }

    private void applySSL() {
        boolean useSSL = idata.getRules().isConditionTrue(SSL_CONDITION);
        String standalonePort = useSSL ? portMap.get(STANDALONE_MANAGEMENT_HTTPS_KEY) : portMap.get(STANDALONE_MANAGEMENT_HTTP_KEY);
        String domainPort = useSSL ? portMap.get(DOMAIN_MANAGEMENT_HTTPS_KEY) : portMap.get(DOMAIN_MANAGEMENT_HTTP_KEY);
        idata.setVariable(STANDALONE_SHORTCUT_KEY, standalonePort);
        idata.setVariable(DOMAIN_SHORTCUT_KEY, domainPort);

        if (useSSL) {
            idata.setVariable(URL_PROTOCOL_KEY, "https");
        }
    }

    /**
     * Parse a string that represents a number into an integer.
     * If we fail to parse the string into an integer return zero
     * @param number String representation of a number
     * @return Integer representation of the string passed in, return 0 if string cannot be parsed into an integer
     */
    private int getNumber(String number) {
        try {
            return Integer.parseInt(number);
        }
        catch (NumberFormatException e) {
            return 0;
        }
    }

    /**
     * Retrieve port number (####) from string in format of ${some.port.reference:####}
     * If the string does not follow the ${<data>} pattern, then return the string passed in.
     * @param portData expected string of format ${some.port.reference:####}, or ####
     * @return string representation of the port (####)
     */
    private String getPortNumber(final String portData) {
        if (portData.startsWith("$") &&  portData.contains(":") &&  portData.endsWith("}")) {
            return portData.substring(portData.indexOf(":") + 1, portData.indexOf("}"));
        }
        else {
            return portData;
        }
    }

    /**
     * There shouldn't be any need for configuration, since this is a dedicated class
     * @param configuration null if no configuration block for the action
     */
    public void initialize(PanelActionConfiguration configuration) {
        // I am the last dinosaur on earth.
    }
}