/*
 * Copyright 2018 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.optaplanner.core.impl.solver.thread;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import org.junit.After;
import org.junit.Test;
import org.optaplanner.core.api.score.buildin.simple.SimpleScore;
import org.optaplanner.core.impl.heuristic.move.DummyMove;
import org.optaplanner.core.impl.heuristic.thread.OrderByMoveIndexBlockingQueue;
import org.optaplanner.core.impl.partitionedsearch.queue.PartitionQueueTest;
import org.optaplanner.core.impl.testdata.domain.TestdataSolution;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.junit.Assert.assertEquals;
import static org.optaplanner.core.impl.testdata.util.PlannerAssert.*;
import static org.optaplanner.core.impl.testdata.util.PlannerAssert.assertSame;
import static org.optaplanner.core.impl.testdata.util.PlannerAssert.fail;

public class OrderByMoveIndexBlockingQueueTest {

    private static final Logger logger = LoggerFactory.getLogger(PartitionQueueTest.class);

    private final ExecutorService executorService = Executors.newFixedThreadPool(2);

    @After
    public void tearDown() throws InterruptedException {
        executorService.shutdownNow();
        if (!executorService.awaitTermination(1, TimeUnit.MILLISECONDS)) {
            logger.warn("Thread pool didn't terminate within the timeout.");
        }
    }

    @Test
    public void addMove() throws InterruptedException {
        // Capacity: 4 moves in circulation + 2 exception handling results
        OrderByMoveIndexBlockingQueue<TestdataSolution> queue = new OrderByMoveIndexBlockingQueue<>(4 + 2);

        queue.startNextStep(0);
        executorService.submit(() -> queue.addMove(0, 0, 0, new DummyMove("a0"), SimpleScore.valueOf(-100)));
        executorService.submit(() -> queue.addMove(1, 0, 1, new DummyMove("a1"), SimpleScore.valueOf(-1000)));
        executorService.submit(() -> queue.addMove(0, 0, 2, new DummyMove("a2"), SimpleScore.valueOf(-200)));
        executorService.submit(() -> queue.addMove(1, 0, 3, new DummyMove("a3"), SimpleScore.valueOf(-30)));
        assertResult("a0", -100, queue.take());
        assertResult("a1", -1000, queue.take());
        assertResult("a2", -200, queue.take());
        executorService.submit(() -> queue.addMove(1, 0, 5, new DummyMove("a5"), SimpleScore.valueOf(-5)));
        executorService.submit(() -> queue.addMove(1, 0, 4, new DummyMove("a4"), SimpleScore.valueOf(-4)));
        assertResult("a3", -30, queue.take());
        executorService.submit(() -> queue.addMove(1, 0, 9, new DummyMove("a9"), SimpleScore.valueOf(-9)));
        assertResult("a4", -4, queue.take());
        assertResult("a5", -5, queue.take());
        executorService.submit(() -> queue.addMove(1, 0, 8, new DummyMove("a8"), SimpleScore.valueOf(-8)));
        executorService.submit(() -> queue.addMove(0, 0, 6, new DummyMove("a6"), SimpleScore.valueOf(-6)));
        executorService.submit(() -> queue.addMove(1, 0, 7, new DummyMove("a7"), SimpleScore.valueOf(-7)));
        assertResult("a6", -6, queue.take());
        executorService.submit(() -> queue.addMove(1, 0, 10, new DummyMove("a10"), SimpleScore.valueOf(-10)));

        queue.startNextStep(1);
        executorService.submit(() -> queue.addMove(0, 1, 0, new DummyMove("b0"), SimpleScore.valueOf(0)));
        executorService.submit(() -> queue.addMove(1, 0, 11, new DummyMove("a11"), SimpleScore.valueOf(-11)));
        assertResult("b0", 0, queue.take());
        executorService.submit(() -> queue.addMove(0, 1, 3, new DummyMove("b3"), SimpleScore.valueOf(-3)));
        executorService.submit(() -> queue.addMove(0, 1, 1, new DummyMove("b1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addMove(0, 1, 2, new DummyMove("b2"), SimpleScore.valueOf(-2)));
        assertResult("b1", -1, queue.take());
        executorService.submit(() -> queue.addMove(0, 1, 4, new DummyMove("b4"), SimpleScore.valueOf(-4)));

        queue.startNextStep(2);
        executorService.submit(() -> queue.addMove(1, 2, 2, new DummyMove("c2"), SimpleScore.valueOf(-2)));
        executorService.submit(() -> queue.addMove(1, 2, 1, new DummyMove("c1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addMove(1, 2, 0, new DummyMove("c0"), SimpleScore.valueOf(0)));
        assertResult("c0", 0, queue.take());
        assertResult("c1", -1, queue.take());
        assertResult("c2", -2, queue.take());
    }

    @Test
    public void addUndoableMove() throws InterruptedException {
        // Capacity: 4 moves in circulation + 2 exception handling results
        OrderByMoveIndexBlockingQueue<TestdataSolution> queue = new OrderByMoveIndexBlockingQueue<>(4 + 2);

        queue.startNextStep(0);
        executorService.submit(() -> queue.addUndoableMove(0, 0, 0, new DummyMove("a0")));
        executorService.submit(() -> queue.addUndoableMove(1, 0, 3, new DummyMove("a3")));
        executorService.submit(() -> queue.addMove(0, 0, 1, new DummyMove("a1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addUndoableMove(1, 0, 2, new DummyMove("a2")));
        assertResult("a0", false, queue.take());
        assertResult("a1", -1, queue.take());
        assertResult("a2", false, queue.take());

        queue.startNextStep(1);
        executorService.submit(() -> queue.addMove(0, 1, 1, new DummyMove("b1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addUndoableMove(1, 0, 4, new DummyMove("a4")));
        executorService.submit(() -> queue.addUndoableMove(1, 1, 0, new DummyMove("b0")));
        assertResult("b0", false, queue.take());
        assertResult("b1", -1, queue.take());
    }

    @Test
    public void addExceptionThrown() throws InterruptedException, ExecutionException {
        // Capacity: 4 moves in circulation + 2 exception handling results
        OrderByMoveIndexBlockingQueue<TestdataSolution> queue = new OrderByMoveIndexBlockingQueue<>(4 + 2);

        queue.startNextStep(0);
        executorService.submit(() -> queue.addMove(0, 0, 1, new DummyMove("a1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addMove(1, 0, 0, new DummyMove("a0"), SimpleScore.valueOf(0)));
        executorService.submit(() -> queue.addMove(0, 0, 2, new DummyMove("a2"), SimpleScore.valueOf(-2)));
        executorService.submit(() -> queue.addMove(1, 0, 3, new DummyMove("a3"), SimpleScore.valueOf(-3)));
        assertResult("a0", 0, queue.take());
        assertResult("a1", -1, queue.take());
        assertResult("a2", -2, queue.take());

        queue.startNextStep(1);
        executorService.submit(() -> queue.addMove(0, 1, 1, new DummyMove("b1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addUndoableMove(1, 0, 4, new DummyMove("a4")));
        executorService.submit(() -> queue.addUndoableMove(1, 1, 0, new DummyMove("b0")));
        IllegalArgumentException exception = new IllegalArgumentException();
        Future<?> exceptionFuture = executorService.submit(() -> queue.addExceptionThrown(1, exception));
        executorService.submit(() -> queue.addMove(0, 1, 2, new DummyMove("b2"), SimpleScore.valueOf(-2)));
        assertResult("b0", false, queue.take());
        assertResult("b1", -1, queue.take());
        exceptionFuture.get(); // Avoid random failing test when the task hasn't started yet
        try {
            queue.take();
            fail("There was no RuntimeException thrown.");
        } catch (RuntimeException e) {
            assertSame(exception, e.getCause());
        }
    }

    @Test
    public void addExceptionIsNotEatenIfNextStepStartsBeforeTaken() throws InterruptedException, ExecutionException {
        // Capacity: 4 moves in circulation + 2 exception handling results
        OrderByMoveIndexBlockingQueue<TestdataSolution> queue = new OrderByMoveIndexBlockingQueue<>(4 + 2);

        queue.startNextStep(0);
        executorService.submit(() -> queue.addMove(0, 0, 1, new DummyMove("a1"), SimpleScore.valueOf(-1)));
        executorService.submit(() -> queue.addMove(1, 0, 0, new DummyMove("a0"), SimpleScore.valueOf(0)));
        executorService.submit(() -> queue.addMove(0, 0, 2, new DummyMove("a2"), SimpleScore.valueOf(-2)));
        executorService.submit(() -> queue.addMove(1, 0, 3, new DummyMove("a3"), SimpleScore.valueOf(-3)));
        IllegalArgumentException exception = new IllegalArgumentException();
        Future<?> exceptionFuture = executorService.submit(() -> queue.addExceptionThrown(1, exception));
        try {
            assertResult("a0", 0, queue.take());
            assertResult("a1", -1, queue.take());
            assertResult("a2", -2, queue.take());

            exceptionFuture.get(); // Avoid random failing test when the task hasn't started yet
            queue.startNextStep(1);
            fail("There was no RuntimeException thrown.");
        } catch (RuntimeException e) {
            assertSame(exception, e.getCause());
        }
    }

    private void assertResult(String moveCode, int score, OrderByMoveIndexBlockingQueue.MoveResult<TestdataSolution> result) {
        assertCode(moveCode, result.getMove());
        assertEquals(SimpleScore.valueOf(score), result.getScore());
    }

    private void assertResult(String moveCode, boolean doable, OrderByMoveIndexBlockingQueue.MoveResult<TestdataSolution> result) {
        assertCode(moveCode, result.getMove());
        assertEquals(doable, result.isMoveDoable());
    }

}
