package com.redhat.installer.installation.processpanel;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class TomcatAddDataSource extends TomcatAddResource {

    /**
     *  Regular expressions to extract server, port and database components from the URL connection for specific DB vendors.
     *  Used to extract and include the properties "serverName", "portNumber" and "databaseName" in the generated resource.
     *  The extraction is enabled by the condition "jdbc.driver.spliturl".
     */
    private static final String DB2_URL_SPLITTER = "jdbc:db2://(?<SERVER>[^:]+):(?<PORT>[^/]+)/(?<DATABASE>[^:]+).*";
    private static final String SYBASE_URL_SPLITTER = "jdbc:sybase:Tds:(?<SERVER>[^:]+):(?<PORT>[^/]+)/(?<DATABASE>[^\\?]+).*";

    public static boolean run(AbstractUIProcessHandler handler, String[] args) {
        idata = AutomatedInstallData.getInstance();
        mHandler = handler;
        List<String> argList= new ArrayList<>(Arrays.asList(args));
        argList.addAll(getCustomProperties());
        argList.addAll(getUrlComponents());
        return addTomcatResource(argList);
    }

    /**
     * Creates a list of custom properties to be added to the datasource
     * @return The list of custom properties
     */
    private static List<String> getCustomProperties() {
        String customProperties = idata.getVariable("jdbc.driver.customproperties");
        if (customProperties != null && !customProperties.isEmpty()) {
            return Arrays.asList(customProperties.split(";"));
        }
        return new ArrayList<>();
    }

    /**
     * Creates a list of properties with the components of the JDBC connection URL.
     * @return The list of URL components
     */
    private static List<String> getUrlComponents() {
        if (!idata.getRules().isConditionTrue("jdbc.driver.spliturl")) {
            return new ArrayList<>();
        }

        switch (idata.getVariable("jdbc.driver.vendor.name")) {
            case "ibmdb2":
                return extractUrlComponentsWith(DB2_URL_SPLITTER);
            case "sybase":
                return extractUrlComponentsWith(SYBASE_URL_SPLITTER);
            default:
                return new ArrayList<>();
        }
    }

    /**
     * Extracts a list of properties with JDBC connection URL components using a regular expression.
     * @param splitter A regular expression used to extract the URL components
     * @return The list of URL components
     */
    private static List<String> extractUrlComponentsWith(String splitter) {
        String connectionUrl = idata.getVariable("jdbc.datasource.connectionurl");
        if (connectionUrl != null && !connectionUrl.isEmpty()) {
            Pattern splitterPattern = Pattern.compile(splitter);
            Matcher matcher = splitterPattern.matcher(connectionUrl);
            if (matcher.find()) {
                return Arrays.asList("serverName=" + matcher.group("SERVER"),
                    "portNumber=" + matcher.group("PORT"),
                    "databaseName=" + matcher.group("DATABASE"));
            }
        }
        return new ArrayList<>();
    }
}
