#!/usr/bin/env bash

set -e

function usage {
    echo "Usage:"
    echo "   create-update.sh -p product_name -c component_list -v version -u updatable_versions -o output_directory"
    echo ""
    echo "   -p, --product <string>             Product name: rhpam or rhdm."
    echo "   -c, --components <list>            Comma separated list of components to include in the change."
    echo "                                      I.e. business-central-eap7-deployable, kie-server-ee8, etc."
    echo "   -v, --version <version>            Current components version."
    echo "   -u, --updatableVersions <list>     List of updatable components versions sorted in ascending order."
    echo "   -o, --outputDirectory <dir>        Output directory to place temporary and update change files."
    echo "   -w, --overwrittenPaths <list>      List of paths that will be forcedly included in the update."
}

while [[ $# -gt 0 ]]
do
    arg="$1"

    case "$arg" in
        -p|--product)
            shift
            PRODUCT="$1"
            ;;
        -c|--components)
            shift
            COMPONENTS="$1"
            ;;
        -v|--version)
            shift
            CURRENT_VERSION="$1"
            ;;
        -u|--updatableVersions)
            shift
            UPDATABLE_VERSIONS="$1"
            ;;
        -o|--outputDirectory)
            shift
            OUTPUT_DIR="$1"
            ;;
        -w|--overwrittenPaths)
            shift
            OVERWRITTEN_PATHS="$1"
            ;;
        *)
            echo "Unknown argument: $1"
            usage
            exit 1
            ;;
    esac
    shift
done

if [[ -z "$PRODUCT" || -z "$COMPONENTS" || -z "$CURRENT_VERSION" || -z "$UPDATABLE_VERSIONS" || -z "$OUTPUT_DIR" ]];
then
    usage
    exit 0
fi

PRODUCT=$(echo "${PRODUCT}" | tr '[:upper:]' '[:lower:]')
COMPONENTS=($(echo "${COMPONENTS}" | tr ',' '\n'))

UPDATABLE_VERSIONS=($(echo "${UPDATABLE_VERSIONS}" | tr ',' '\n'))
ALL_VERSIONS=("${UPDATABLE_VERSIONS[@]}" "${CURRENT_VERSION}")

OVERWRITTEN_PATHS=($(echo "${OVERWRITTEN_PATHS}" | tr ',' '\n'))

COMPONENTS_DIR="${OUTPUT_DIR}/components"
CHECKSUMS_DIR="${OUTPUT_DIR}/checksums"
CHANGES_DIR="${OUTPUT_DIR}/changes"
NEW_CONTENT_DIR="${CHANGES_DIR}/new-content"

# Cleans up the output directory and recreates the directory tree to store working data and the update changes
function prepare-workspace {
    echo "Preparing workspace ..."
    rm -rf "${COMPONENTS_DIR}" "${CHECKSUMS_DIR}" "${CHANGES_DIR}"
    mkdir -p "${NEW_CONTENT_DIR}"
    for version in "${ALL_VERSIONS[@]}"
    do
        mkdir -p "${CHECKSUMS_DIR}/${version}"
    done
    for component in ${COMPONENTS[@]}
    do
        mkdir -p "${CHANGES_DIR}/${component}"
    done
}

# Downloads and unpack using Maven the component binaries to the output directory
function unpack-dependencies {
    for version in "${ALL_VERSIONS[@]}"
    do
        echo "Unpacking dependencies for version ${version} ..."
        for component in ${COMPONENTS[@]}
        do
            componentDir="${COMPONENTS_DIR}/${version}/${component}"
            mvn dependency:unpack -Dartifact="org.kie.rhba:${PRODUCT}:${version}:zip:${component}" -DoutputDirectory="${componentDir}"
        done
    done
}

# Removes the EAP versioning from directories name
function remove-eap-versioning {
    echo "Stripping out EAP versioning from directories ..."
    find "${COMPONENTS_DIR}" -type d -name "jboss-eap-*" -printf "%p %h/jboss-eap\n" | xargs -rt -n2 mv
}

# Computes the MD5 checksums for every file in a component and version
function compute-checksums {
    for component in ${COMPONENTS[@]}
    do
        echo "Computing checksums for ${component} ..."
        for version in "${ALL_VERSIONS[@]}"
        do
            componentDir="${COMPONENTS_DIR}/${version}/${component}"
            for file in $(find ${componentDir} -type f -printf '%P\n' | sort)
            do
                checksum=$(md5sum "${componentDir}/${file}" | awk '{print $1}')
                echo "${file}=${checksum}" >> "${CHECKSUMS_DIR}/${version}/${component}.txt"
            done
        done
        cp "${CHECKSUMS_DIR}/${CURRENT_VERSION}/${component}.txt" "${CHANGES_DIR}/${component}/checksums.txt"
    done
}

# Aggregates the MD5 checksums by component for every version
function aggregate-checksums {
    for component in ${COMPONENTS[@]}
    do
        echo "Aggregating checksums for ${component} ..."
        aggregatedChecksums="${CHANGES_DIR}/${component}/checksums.txt"
        for version in "${ALL_VERSIONS[@]}"
        do
            prevVersionChecksums="${CHECKSUMS_DIR}/${version}/${component}.txt"
            cat "${prevVersionChecksums}" >> "${aggregatedChecksums}"
        done
        cat "${aggregatedChecksums}" | sort -u > "${aggregatedChecksums}.tmp"
        mv -f "${aggregatedChecksums}.tmp" "${aggregatedChecksums}"
    done
}

# Computes the new content per component as the difference between the $CURRENT_VERSION and every $UPDATABLE_VERSIONS
function compute-contents-to-add {
    for component in ${COMPONENTS[@]}
    do
        echo "Computing contents to add to ${component} ..."
        currentVersionChecksums="${CHECKSUMS_DIR}/${CURRENT_VERSION}/${component}.txt"
        newContent="${CHANGES_DIR}/${component}/new-content.txt"
        tempNewContent="${newContent}.tmp"
        for version in "${UPDATABLE_VERSIONS[@]}"
        do
            prevVersionChecksums="${CHECKSUMS_DIR}/${version}/${component}.txt"
            grep -Fxv -f "${prevVersionChecksums}" "${currentVersionChecksums}" >> "${tempNewContent}"
        done
        for path in ${OVERWRITTEN_PATHS[@]}
        do
            grep -E "^${path}.*$" "${currentVersionChecksums}" >> "${tempNewContent}" || true
        done
        # format from "{path}={md5sum}" to "{path}={filename}.{md5sum}"
        sort -u "${tempNewContent}" | sed 's/\(.*\/\)\(.*\)=/\1\2=\2./g' > "${newContent}"
        rm -f "${tempNewContent}"
    done
}

# Computes the contents to be removed by component as the aggregation of all $UPDATABLE_VERSIONS minus the $CURRENT_VERSION
function compute-contents-to-remove {
    for component in ${COMPONENTS[@]}
    do
        echo "Computing contents to remove from ${component} ..."
        aggregatedChecksums="${CHANGES_DIR}/${component}/checksums.txt"
        currentVersionChecksums="${CHECKSUMS_DIR}/${CURRENT_VERSION}/${component}.txt"
        removeList="${CHANGES_DIR}/${component}/remove-list.txt"
        tempRemoveList="${removeList}.tmp"
        grep -Fxv -f "${currentVersionChecksums}" "${aggregatedChecksums}" > "${removeList}"
        awk -F '=' '{print $1}' "${removeList}" > "${tempRemoveList}"
        sed -E 's/-redhat-[0-9]+\.jar/*.jar/g' "${tempRemoveList}" | sort -u > "${removeList}"
        rm -f "${tempRemoveList}"
    done
}

# Copies all the components' new files to a common place to avoid repeating them per component
function aggregate-contents-to-add {
    for component in ${COMPONENTS[@]}
    do
        echo "Aggregating contents for ${component} ..."
        currentVersionComponentsDir="${COMPONENTS_DIR}/${CURRENT_VERSION}/${component}"
        componentNewContent="${CHANGES_DIR}/${component}/new-content.txt"
        while IFS= read -r newContent;
        do
            filePath="${newContent/=*/}"
            fileName="${newContent/*=/}"
            if [[ -f "${currentVersionComponentsDir}/${filePath}" && ! -f "${NEW_CONTENT_DIR}/${fileName}" ]]
            then
                cp "${currentVersionComponentsDir}/${filePath}" "${NEW_CONTENT_DIR}/${fileName}"
            fi
        done < "${componentNewContent}"
    done
}

prepare-workspace
unpack-dependencies
remove-eap-versioning
compute-checksums
aggregate-checksums
compute-contents-to-add
compute-contents-to-remove
aggregate-contents-to-add
