package org.jboss.brmsbpmsuite.patching.systemproperty;

import org.apache.commons.io.FileUtils;
import org.jboss.brmsbpmsuite.patching.client.ConfigurationDiffsGeneratorTest;
import org.jboss.brmsbpmsuite.patching.xml.XMLDocumentDao;
import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.core.Is.is;

public class SystemPropertyPatcherTest {

    private static final File TEMP_DIR = new File("./target/tmp-tests-dir");
    private static final File PATCHES_DIR = createFileFromResource("/config-test/updates");
    private static final File SERVER_CONFIG_DIR = createFileFromResource("/config-test/eap-servers/eap-version-1/standalone/configuration");

    private XMLDocumentDao documentDao;

    @Before
    public void setUp() throws ParserConfigurationException, TransformerConfigurationException, IOException {
        documentDao = new XMLDocumentDao();
        TEMP_DIR.mkdirs();
        FileUtils.copyDirectory(SERVER_CONFIG_DIR, TEMP_DIR);
    }

    @Test
    public void shouldApplyPropertyAddition() throws IOException, SAXException, ParserConfigurationException {
        Document baseDocument = loadDocument(TEMP_DIR, "standalone-1.xml");
        Document updateWithAdd = loadDocument(PATCHES_DIR, "standalone-1.xml");

        Document updatedDocument = new SystemPropertyPatcher(baseDocument).apply(updateWithAdd);

        NodeList properties = updatedDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(3));

        NamedNodeMap addedProperty = properties.item(2).getAttributes();
        assertThat(addedProperty.getNamedItem("name").getNodeValue(), is("property3"));
        assertThat(addedProperty.getNamedItem("value").getNodeValue(), is("value3"));
    }

    @Test
    public void shouldApplyPropertyValueModification() throws IOException, SAXException, ParserConfigurationException {
        Document baseDocument = loadDocument(TEMP_DIR, "standalone-2.xml");
        Document updateWithModification = loadDocument(PATCHES_DIR, "standalone-2.xml");

        Document updatedDocument = new SystemPropertyPatcher(baseDocument).apply(updateWithModification);

        NodeList properties = updatedDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(2));

        NamedNodeMap modifiedProperty = properties.item(1).getAttributes();
        assertThat(modifiedProperty.getNamedItem("name").getNodeValue(), is("property2"));
        assertThat(modifiedProperty.getNamedItem("value").getNodeValue(), is("different-value2"));
    }

    @Test
    public void shouldApplyPropertyRemoval() throws ParserConfigurationException, IOException, SAXException {
        Document baseDocument = loadDocument(TEMP_DIR, "standalone-3.xml");
        Document updateWithRemoval = loadDocument(PATCHES_DIR, "standalone-3.xml");

        Document updatedDocument = new SystemPropertyPatcher(baseDocument).apply(updateWithRemoval);

        NodeList properties = updatedDocument.getDocumentElement().getElementsByTagName("property");
        for (int i = 0; i < properties.getLength(); i++) {
            NamedNodeMap property = properties.item(i).getAttributes();
            assertThat(property.getNamedItem("name").getNodeValue(), is(not("property2")));
        }
    }

    @Test
    public void shouldApplyPropertyAdditionOnDocumentsWithoutSystemPropertiesNode() throws IOException, SAXException, ParserConfigurationException {
        Document baseDocument = loadDocument(TEMP_DIR, "standalone.no-properties.xml");
        Document updateWithAdd = loadDocument(PATCHES_DIR, "standalone-1.xml");

        Document updatedDocument = new SystemPropertyPatcher(baseDocument).apply(updateWithAdd);

        NodeList properties = updatedDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(1));

        NamedNodeMap addedProperty = properties.item(0).getAttributes();
        assertThat(addedProperty.getNamedItem("name").getNodeValue(), is("property3"));
        assertThat(addedProperty.getNamedItem("value").getNodeValue(), is("value3"));
    }

    @Test
    public void shouldApplyPropertyValueModificationAsAdditionOnDocumentsWithoutSystemPropertiesNode() throws IOException, SAXException, ParserConfigurationException {
        Document baseDocument = loadDocument(TEMP_DIR, "standalone.no-properties.xml");
        Document updateWithModification = loadDocument(PATCHES_DIR, "standalone-2.xml");

        Document updatedDocument = new SystemPropertyPatcher(baseDocument).apply(updateWithModification);

        NodeList properties = updatedDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(1));

        NamedNodeMap modifiedProperty = properties.item(0).getAttributes();
        assertThat(modifiedProperty.getNamedItem("name").getNodeValue(), is("property2"));
        assertThat(modifiedProperty.getNamedItem("value").getNodeValue(), is("different-value2"));
    }

    @Test
    public void shouldNotApplyPropertyRemovalOnDocumentsWithoutSystemPropertiesNode() throws ParserConfigurationException, IOException, SAXException {
        Document baseDocument = loadDocument(TEMP_DIR, "standalone.no-properties.xml");
        Document updateWithRemoval = loadDocument(PATCHES_DIR, "standalone-4.xml");

        Document updatedDocument = new SystemPropertyPatcher(baseDocument).apply(updateWithRemoval);

        NodeList properties = updatedDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(0));
    }

    private static File createFileFromResource(String filePath) {
        return new File(ConfigurationDiffsGeneratorTest.class.getResource(filePath).getPath());
    }

    private Document loadDocument(File dir, String filename) throws IOException, SAXException {
        File file = new File(dir, filename);
        return documentDao.load(new FileInputStream(file));
    }

}