/*
 * #%L
 * Wildfly Camel :: Testsuite
 * %%
 * Copyright (C) 2013 - 2014 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

package org.wildfly.camel.test.jacksonxml;

import org.apache.camel.CamelContext;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.jacksonxml.JacksonXMLDataFormat;
import org.apache.camel.impl.DefaultCamelContext;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.modules.ModuleClassLoader;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.wildfly.camel.test.common.types.Customer;
import org.wildfly.extension.camel.CamelAware;

@CamelAware
@RunWith(Arquillian.class)
public class JacksonXMLIntegrationTest {

    static String XML_STRING = "<Customer><firstName>John</firstName><lastName>Doe</lastName></Customer>";

    @Deployment
    public static JavaArchive deployment() {
        final JavaArchive archive = ShrinkWrap.create(JavaArchive.class, "jacksonxml-tests");
        archive.addClasses(Customer.class);
        return archive;
    }

    @Test
    public void testClassLoader() throws Exception {

        ClassLoader componentLoader = JacksonXMLDataFormat.class.getClassLoader();
        Assert.assertTrue("Not a ModuleClassLoader: " + componentLoader, componentLoader instanceof ModuleClassLoader);
        Assert.assertEquals("org.apache.camel.component.jacksonxml", ((ModuleClassLoader) componentLoader).getModule().getName());

        ClassLoader moduleLoader = componentLoader.loadClass("com.fasterxml.jackson.annotation.JsonIgnoreProperties").getClassLoader();
        Assert.assertTrue("Not a ModuleClassLoader: " + moduleLoader, moduleLoader instanceof ModuleClassLoader);
        Assert.assertEquals("com.fasterxml.jackson.core.jackson-annotations", ((ModuleClassLoader) moduleLoader).getModule().getName());

        moduleLoader = componentLoader.loadClass("com.fasterxml.jackson.databind.ObjectMapper").getClassLoader();
        Assert.assertTrue("Not a ModuleClassLoader: " + moduleLoader, moduleLoader instanceof ModuleClassLoader);
        Assert.assertEquals("com.fasterxml.jackson.core.jackson-databind", ((ModuleClassLoader) moduleLoader).getModule().getName());

        ClassLoader deploymentLoader = getClass().getClassLoader();

        moduleLoader = deploymentLoader.loadClass("com.fasterxml.jackson.annotation.JsonIgnoreProperties").getClassLoader();
        Assert.assertTrue("Not a ModuleClassLoader: " + moduleLoader, moduleLoader instanceof ModuleClassLoader);
        Assert.assertEquals("com.fasterxml.jackson.core.jackson-annotations", ((ModuleClassLoader) moduleLoader).getModule().getName());

        moduleLoader = deploymentLoader.loadClass("com.fasterxml.jackson.databind.ObjectMapper").getClassLoader();
        Assert.assertTrue("Not a ModuleClassLoader: " + moduleLoader, moduleLoader instanceof ModuleClassLoader);
        Assert.assertEquals("com.fasterxml.jackson.core.jackson-databind", ((ModuleClassLoader) moduleLoader).getModule().getName());
    }

    @Test
    public void testMarshal() throws Exception {

        CamelContext camelctx = new DefaultCamelContext();
        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:start")
                .marshal().jacksonxml();
            }
        });

        camelctx.start();
        try {
            ProducerTemplate producer = camelctx.createProducerTemplate();
            String result = producer.requestBody("direct:start", new Customer("John", "Doe"), String.class);
            Assert.assertEquals(XML_STRING, result);
        } finally {
            camelctx.stop();
        }
    }

    @Test
    public void testUnmarshal() throws Exception {

        CamelContext camelctx = new DefaultCamelContext();
        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:start")
                .unmarshal().jacksonxml(Customer.class);
            }
        });

        camelctx.start();
        try {
            ProducerTemplate producer = camelctx.createProducerTemplate();
            Customer customer = producer.requestBody("direct:start", XML_STRING, Customer.class);
            Assert.assertEquals("John", customer.getFirstName());
            Assert.assertEquals("Doe", customer.getLastName());
        } finally {
            camelctx.stop();
        }
    }
}
