/**
 * Copyright 2016 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package io.fabric8.maven.core.handler;

import io.fabric8.kubernetes.api.model.Probe;
import io.fabric8.maven.core.config.ProbeConfig;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

public class ProbeHandlerTest {
    Probe probe;
    ProbeHandler probeHandler = new ProbeHandler();

    ProbeConfig probeConfig;

    @Test
    public void getProbeEmptyTest() {
        //EmptyProbeConfig

        probeConfig = null;

        probe = probeHandler.getProbe(probeConfig);

        assertNull(probe);
    }

    @Test
    public void getProbeNullTest() {
        //ProbeConfig without any action

        probeConfig = new ProbeConfig.Builder()
                .build();

        probe = probeHandler.getProbe(probeConfig);

        assertNull(probe);
    }

    @Test
    public void getHTTPProbeWithEmptyURLTest() {
        //ProbeConfig with HTTPGet Action

        //withEmptyUrl
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).getUrl(null)
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNull(probe);
    }

    @Test
    public void getHTTPProbeWithHTTPURLTest() {

        //ProbeConfig with HTTPGet Action
        //withUrl
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).getUrl("http://www.healthcheck.com:8080/healthz")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
        assertEquals("www.healthcheck.com",probe.getHttpGet().getHost());
        assertEquals(null,probe.getHttpGet().getHttpHeaders());
        assertEquals("/healthz",probe.getHttpGet().getPath());
        assertEquals(8080,probe.getHttpGet().getPort().getIntVal().intValue());
        assertEquals("HTTP",probe.getHttpGet().getScheme());
        assertNull(probe.getExec());
        assertNull(probe.getTcpSocket());
    }

    @Test
    public void getHTTPProbeWithoutHTTPURLTest() {
        //ProbeConfig with HTTPGet Action
        //URL Without http Portocol
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).getUrl("www.healthcheck.com:8080/healthz")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        assertNull(probe);
    }

    @Test(expected = IllegalArgumentException.class)
    public void getHTTPProbeWithInvalidURLTest() {
        //ProbeConfig with HTTPGet Action
        //withInvalidUrl
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).getUrl("httphealthcheck.com:8080/healthz")
                .build();

        probe = probeHandler.getProbe(probeConfig);
    }

    @Test
    public void getExecProbeWithEmptyExecTest() {
        //ProbeConfig with Exec Action
        //withEmptyExec
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).exec("")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNull(probe);
    }

    @Test
    public void getExecProbeWithExecTest() {
        //ProbeConfig with Exec Action
        //withExec
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).exec("cat /tmp/probe")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
        assertNotNull(probe.getExec());
        assertEquals(2,probe.getExec().getCommand().size());
        assertEquals("[cat, /tmp/probe]",probe.getExec().getCommand().toString());
        assertNull(probe.getHttpGet());
        assertNull(probe.getTcpSocket());
    }

    @Test
    public void getExecProbeWithInvalidExecTest() {
        //ProbeConfig with Exec Action
        //withInvalidExec
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).exec("   ")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNull(probe);
    }

    @Test
    public void getTCPProbeWithoutURLTest() {
        //ProbeConfig with TCP Action
        //withno url, only port
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).tcpPort("80")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertNull(probe.getHttpGet());
        assertNotNull(probe.getTcpSocket());
        assertEquals(80,probe.getTcpSocket().getPort().getIntVal().intValue());
        assertNull(probe.getTcpSocket().getHost());
        assertNull(probe.getExec());
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
    }

    @Test
    public void getTCPProbeWithHTTPURLAndPortTest() {
        //ProbeConfig with TCP Action
        //withport and url but with http request
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5)
                .getUrl("http://www.healthcheck.com:8080/healthz").tcpPort("80")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertNotNull(probe.getHttpGet());
        assertNull(probe.getTcpSocket());
        assertNull(probe.getExec());
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
        assertEquals("www.healthcheck.com",probe.getHttpGet().getHost());
        assertEquals(null,probe.getHttpGet().getHttpHeaders());
        assertEquals("/healthz",probe.getHttpGet().getPath());
        assertEquals(8080,probe.getHttpGet().getPort().getIntVal().intValue());
        assertEquals("HTTP",probe.getHttpGet().getScheme());
    }

    @Test
    public void getTCPProbeWithNonHTTPURLTest() {
        //ProbeConfig with TCP Action
        //withport and url but with other request and port as int
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5)
                .failureThreshold(3).successThreshold(1)
                .getUrl("tcp://www.healthcheck.com:8080/healthz").tcpPort("80")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertNull(probe.getHttpGet());
        assertNotNull(probe.getTcpSocket());
        assertNull(probe.getExec());
        assertEquals(80,probe.getTcpSocket().getPort().getIntVal().intValue());
        assertEquals("www.healthcheck.com",probe.getTcpSocket().getHost());
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
        assertEquals(3, probe.getFailureThreshold().intValue());
        assertEquals(1, probe.getSuccessThreshold().intValue());
    }

    @Test
    public void getTCPProbeWithNonHTTPURLAndStringPortTest() {
        //ProbeConfig with TCP Action
        //withport and url but with other request and port as string
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5)
                .getUrl("tcp://www.healthcheck.com:8080/healthz").tcpPort("httpPort")
                .successThreshold(1)
                .failureThreshold(3)
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertNull(probe.getHttpGet());
        assertNotNull(probe.getTcpSocket());
        assertNull(probe.getExec());
        assertEquals("httpPort",probe.getTcpSocket().getPort().getStrVal());
        assertEquals("www.healthcheck.com",probe.getTcpSocket().getHost());
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
        assertEquals(3, probe.getFailureThreshold().intValue());
        assertEquals(1, probe.getSuccessThreshold().intValue());
    }

    @Test
    public void getTCPWithHTTPURLAndWithoutPort() {
        //ProbeConfig with TCP Action
        //without port and url with http request
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5)
                .getUrl("http://www.healthcheck.com:8080/healthz")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNotNull(probe);
        assertNotNull(probe.getHttpGet());
        assertNull(probe.getTcpSocket());
        assertNull(probe.getExec());
        assertEquals(5,probe.getInitialDelaySeconds().intValue());
        assertEquals(5,probe.getTimeoutSeconds().intValue());
        assertEquals("www.healthcheck.com",probe.getHttpGet().getHost());
        assertEquals(null,probe.getHttpGet().getHttpHeaders());
        assertEquals("/healthz",probe.getHttpGet().getPath());
        assertEquals(8080,probe.getHttpGet().getPort().getIntVal().intValue());
        assertEquals("HTTP",probe.getHttpGet().getScheme());
    }

    @Test
    public void getTCPProbeWithTCPURLTest() {
        //ProbeConfig with TCP Action
        //without port and url with tcp request
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5)
                .getUrl("tcp://www.healthcheck.com:8080/healthz")
                .build();

        probe = probeHandler.getProbe(probeConfig);
        //assertion
        assertNull(probe);
    }

    @Test(expected = IllegalArgumentException.class)
    public void getTCPProbeWithInvalidURLTest() {
        //ProbeConfig with TCP Action
        //withInvalidUrl
        probeConfig = new ProbeConfig.Builder()
                .initialDelaySeconds(5).timeoutSeconds(5).getUrl("healthcheck.com:8080/healthz")
                .tcpPort("80")
                .build();

        probe = probeHandler.getProbe(probeConfig);
    }
}
