/*
 * #%L
 * Wildfly Camel :: Testsuite
 * %%
 * Copyright (C) 2013 - 2015 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package org.wildfly.camel.test.cxf.rs;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.core.MediaType;

import org.apache.camel.CamelContext;
import org.apache.camel.ServiceStatus;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wildfly.camel.test.common.http.HttpRequest;
import org.wildfly.camel.test.common.http.HttpRequest.HttpResponse;
import org.wildfly.camel.test.common.types.GreetingService;
import org.wildfly.camel.test.common.utils.ManifestBuilder;
import org.wildfly.camel.utils.IllegalArgumentAssertion;
import org.wildfly.extension.camel.CamelAware;
import org.wildfly.extension.camel.CamelContextRegistry;

@CamelAware
@RunWith(Arquillian.class)
public class CXFRSSecureConsumerIntegrationTest {

    static final Logger LOG = LoggerFactory.getLogger(CXFRSSecureConsumerIntegrationTest.class);
    
    private static String RS_ENDPOINT_PATH = "/cxfrestful/greet/hello/Kermit";
    private static String INSECURE_RS_ENDPOINT = "http://localhost:8080" + RS_ENDPOINT_PATH;
    private static String SECURE_RS_ENDPOINT = "https://localhost:8443" + RS_ENDPOINT_PATH;

    @ArquillianResource
    CamelContextRegistry contextRegistry;

    @Deployment
    public static WebArchive deployment() {
        return ShrinkWrap.create(WebArchive.class, "cxfrs-secure-consumer-tests.war")
            .addClasses(GreetingService.class, HttpRequest.class)
            .addAsResource("cxf/spring/cxfrs-secure-consumer-camel-context.xml", "cxfrs-secure-consumer-camel-context.xml")
            .setManifest(() -> {
                ManifestBuilder builder = new ManifestBuilder();
                builder.addManifestHeader("Dependencies", "org.jboss.resteasy.resteasy-jaxrs");
                return builder.openStream();
            });
    }

    @Test
    public void testCXFSecureConsumer() throws Exception {

        // Force WildFly to generate a self-signed SSL cert & keystore
        HttpRequest.get("https://localhost:8443").throwExceptionOnFailure(false).getResponse();

        // Use the generated keystore
        Path keystorePath = Paths.get(System.getProperty("jboss.server.config.dir"), "application.keystore");
        Assert.assertTrue("Is file: " + keystorePath, keystorePath.toAbsolutePath().toFile().isFile());
        
        SSLContext sslContext = new SSLContextBuilder()
                .keystorePath(keystorePath)
                .keystorePassword("password")
                .build();
        
        SSLContext.setDefault(sslContext);
        
        CamelContext camelctx = contextRegistry.getCamelContext("cxfrs-secure-undertow");
        Assert.assertNotNull("Expected cxfrs-secure-undertow to not be null", camelctx);
        Assert.assertEquals(ServiceStatus.Started, camelctx.getStatus());

        ClassLoader tccl = Thread.currentThread().getContextClassLoader();
        try {
            // The JAXRS Client API needs to see resteasy
            Thread.currentThread().setContextClassLoader(getClass().getClassLoader());

            Client client = ClientBuilder.newBuilder().sslContext(sslContext).build();

            Object result = client.target(SECURE_RS_ENDPOINT).request(MediaType.APPLICATION_JSON).get(String.class);
            Assert.assertEquals("Hello Kermit", result);

            // Verify that if we attempt to use HTTP, we get a 302 redirect to the HTTPS endpoint URL
            HttpResponse response = HttpRequest.get(INSECURE_RS_ENDPOINT)
                .throwExceptionOnFailure(false)
                .followRedirects(false)
                .getResponse();
            
            Assert.assertEquals(302, response.getStatusCode());
            Assert.assertEquals(response.getHeader("Location"), SECURE_RS_ENDPOINT);
            
        } finally {
            System.clearProperty("javax.net.ssl.trustStorePassword");
            System.clearProperty("javax.net.ssl.trustStore");
            Thread.currentThread().setContextClassLoader(tccl);
        }
    }

    /**
     * Borrowed from io.nessus.common.rest.SSLContextBuilder
     * 
     * https://github.com/tdiesler/nessus-common/blob/master/rest/src/main/java/io/nessus/common/rest/SSLContextBuilder.java
     */
    static class SSLContextBuilder {

        private String keystoreType = KeyStore.getDefaultType();
        private char[] keystorePassword = "changeit".toCharArray();
        private Path keystorePath;
        
        public SSLContextBuilder keystorePath(Path keysPath) {
            this.keystorePath = keysPath;
            return this;
        }
        
        public SSLContextBuilder keystorePassword(String keysPassword) {
            this.keystorePassword = keysPassword.toCharArray();
            return this;
        }
        
        public SSLContext build() throws IOException, GeneralSecurityException {
            
            KeyStore keystore = loadKeyStore(keystorePath, keystoreType, keystorePassword);
            
            SSLContext sslContext;
            try {
                
                KeyManager[] keyManagers = buildKeyManagers(keystore, keystorePassword);
                TrustManager[] trustManagers = buildTrustManagers(keystore);
                
                sslContext = SSLContext.getInstance("TLS");
                sslContext.init(keyManagers, trustManagers, null);
            }
            catch (NoSuchAlgorithmException | KeyManagementException ex) {
                throw new IOException("Unable to create and initialise the SSLContext", ex);
            }

            return sslContext;
        }

        private KeyStore loadKeyStore(Path keystorePath, String keystoreType, char[] keystorePassword) throws IOException, GeneralSecurityException {
            IllegalArgumentAssertion.assertNotNull(keystorePath, "Null keystorePath");
            IllegalArgumentAssertion.assertNotNull(keystoreType, "Null keystoreType");
            IllegalArgumentAssertion.assertNotNull(keystorePassword, "Null keysPassword");
            
            KeyStore keystore = KeyStore.getInstance(keystoreType);
            
            LOG.info("Loading keystore file: {}", keystorePath);
            
            try (FileInputStream fis = new FileInputStream(keystorePath.toFile())) {
                keystore.load(fis, keystorePassword);
            }
            
            return keystore;
        }

        private KeyManager[] buildKeyManagers(final KeyStore keyStore, char[] keysPassword) throws GeneralSecurityException  {
            String keyAlgorithm = KeyManagerFactory.getDefaultAlgorithm();
            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(keyAlgorithm);
            keyManagerFactory.init(keyStore, keysPassword);
            KeyManager[] keyManagers = keyManagerFactory.getKeyManagers();
            return keyManagers;
        }

        private TrustManager[] buildTrustManagers(final KeyStore trustStore) throws IOException, GeneralSecurityException {
            String trustAlgorithm = KeyManagerFactory.getDefaultAlgorithm();
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(trustAlgorithm);
            trustManagerFactory.init(trustStore);
            TrustManager[] trustManagers = trustManagerFactory.getTrustManagers();
            return trustManagers;
        }
    }
}
