/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2011, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.test.iiopssl.basic;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.OperateOnDeployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.container.test.api.TargetsContainer;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.as.arquillian.container.NetworkUtils;
import org.jboss.as.test.shared.FileUtils;
import org.jboss.as.test.shared.PropertiesValueResolver;
import org.jboss.as.test.shared.integration.ejb.security.Util;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.omg.CORBA.*;
import org.omg.CORBA.Object;
import org.omg.CosNaming.NamingContextExt;
import org.omg.CosNaming.NamingContextExtHelper;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.login.LoginException;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Properties;

/**
 * A simple IIOP over SSL invocation for one AS7 server to another
 * There's 3 servers involved in the test:
 * <ul>
 *     <li>iiop-ssl-server hosts remote EJB and requires SSL for ORB calls</li>
 *     <li>iiop-ssl-client hosts client EJB and provides SSL configuration</li>
 *     <li>iiop-non-ssl-client hosts client EJB but doesn't provides SSL configuration</li>
 * </ul>
 *
 */
@RunWith(Arquillian.class)
public class IIOPSslInvocationTestCase {

    private static final int SERVER_SSL_PORT = 3729;
    private static final int SERVER_NONSSL_PORT = 3728;
    private static final String SERVER_NODE = "node1";
    private static final String SERVER_HOST = NetworkUtils.formatPossibleIpv6Address((String) System.getProperty(SERVER_NODE));

    private static final String SSL_CORBALOC = String.format("corbaloc:ssliop:1.2@%s:%d/JBoss/Naming/root", SERVER_HOST, SERVER_SSL_PORT);
    private static final String NONSSL_CORBALOC = String.format("corbaloc:ssliop:1.2@%s:%d/JBoss/Naming/root", SERVER_HOST, SERVER_NONSSL_PORT);

    @Deployment(name = "server", testable = false)
    @TargetsContainer("iiop-ssl-server")
    public static Archive<?> deployment() {
        final JavaArchive jar = ShrinkWrap.create(JavaArchive.class, "server.jar");
        jar.addClasses(IIOPSslStatelessBean.class, IIOPSslStatelessHome.class, IIOPSslStatelessRemote.class)
                .addAsManifestResource(IIOPSslInvocationTestCase.class.getPackage(), "jboss-ejb3.xml", "jboss-ejb3.xml");
        return jar;
    }

    @Deployment(name = "ssl-client", testable = true)
    @TargetsContainer("iiop-ssl-client")
    public static Archive<?> clientDeployment() {

        /*
         * The @EJB annotation doesn't allow to specify the address dynamically. So, istead of
         *       @EJB(lookup = "corbaname:iiop:localhost:3628#IIOPTransactionalStatelessBean")
         *       private IIOPTransactionalHome home;
         * we need to do this trick to get the ${node0} sys prop into ejb-jar.xml
         * and have it injected that way.
         */
        String ejbJar = FileUtils.readFile(IIOPSslInvocationTestCase.class, "ejb-jar.xml");

        final Properties properties = new Properties();
        properties.put("node1", SERVER_HOST);
        properties.put("port", String.valueOf(SERVER_SSL_PORT));

        final JavaArchive jar = ShrinkWrap.create(JavaArchive.class, "client.jar");
        jar.addClasses(ClientEjb.class, IIOPSslStatelessHome.class, IIOPSslStatelessRemote.class, IIOPSslInvocationTestCase.class, Util.class)
                .addAsManifestResource(IIOPSslInvocationTestCase.class.getPackage(), "jboss-ejb3.xml", "jboss-ejb3.xml")
                .addAsManifestResource(new StringAsset(PropertiesValueResolver.replaceProperties(ejbJar, properties)), "ejb-jar.xml");
        return jar;
    }

    @Deployment(name = "non-ssl-client", testable = true)
    @TargetsContainer("iiop-non-ssl-client")
    public static Archive<?> clientDeployment2() {
        final JavaArchive jar = ShrinkWrap.create(JavaArchive.class, "client.jar");
        jar.addClasses(ClientEjb.class, IIOPSslStatelessHome.class, IIOPSslStatelessRemote.class, IIOPSslInvocationTestCase.class, Util.class)
                .addAsManifestResource(IIOPSslInvocationTestCase.class.getPackage(), "jboss-ejb3.xml", "jboss-ejb3.xml")
                .addAsManifestResource(IIOPSslInvocationTestCase.class.getPackage(), "ejb-jar-without-injection.xml", "ejb-jar.xml");
        return jar;
    }


    // Directly using ORB to avoid setting up InitialContext and ORBSingleton
    @Test
    @RunAsClient
    public void testSuccesfulJacorbExecutionOverSSLPort() throws Exception {
        ORB orb = null;
        Object beanHome = null;
        Object bean = null;
        try {
            orb = ORB.init(new String[]{"-ORBInitRef", "NameService=" + SSL_CORBALOC}, orbProperties(true));

            final Object object = orb.resolve_initial_references("NameService");
            final NamingContextExt nce = NamingContextExtHelper.narrow(object);
            beanHome = nce.resolve_str("IIOPSslStatelessBean");
            bean = createBean(orb, beanHome);

            int result = callHello(orb, bean);
            Assert.assertEquals(1, result);

        } finally {
            if (beanHome != null)
                beanHome._release();
            if (bean != null)
                bean._release();
            if (orb != null)
                orb.destroy();

        }
    }



    @Test
    @RunAsClient
    public void testFailedLookupOverNonSSLPort() throws Exception {
        ORB orb = null;
        try {
            orb = ORB.init(new String[]{"-ORBInitRef", "NameService=" + NONSSL_CORBALOC}, orbProperties(false));

            final Object object = orb.resolve_initial_references("NameService");
            final NamingContextExt nce = NamingContextExtHelper.narrow(object);

            Assert.fail("The non-SSL port should not be available");
        } catch (org.omg.CORBA.TRANSIENT e) {
            // OK - failed to connect to non-SSL port
        } finally {
            if (orb != null)
                orb.shutdown(false);
        }
    }

    // tests below work on deployments checking connection between two servers

    @Test
    @OperateOnDeployment("ssl-client")
    public void testInjectRemoteEjbUsingSsl() throws IOException, NamingException, LoginException {
        final ClientEjb ejb = client();
        Assert.assertEquals(1, ejb.getRemoteMessage());
    }

    @Test
    @OperateOnDeployment("ssl-client")
    public void testCallRemoteEjbUsingSSLPort() throws Exception {
        final ClientEjb ejb = client();
        Assert.assertEquals(1, ejb.lookupSsl(SERVER_HOST, SERVER_SSL_PORT));
    }

    @Test
    @OperateOnDeployment("non-ssl-client")
    public void failWhenDoingManualLookupUsingNonSSLPort() throws Exception {
        final ClientEjb ejb = client();
        try {
            ejb.lookup(SERVER_HOST, SERVER_NONSSL_PORT);
            Assert.fail("The lookup using non-SSL port should not be possible");
        } catch (NamingException e) {
            Assert.assertTrue("Unexpected error: expected [ " + String.format("Retries exceeded, couldn't reconnect to %s:%d", SERVER_HOST, SERVER_NONSSL_PORT) + " ], but got[ " + e.getMessage() + "]",
                    e.getMessage().startsWith("Retries exceeded, couldn't reconnect to"));
        }
    }

    private int callHello(ORB orb, Object bean) {
        Request helloRequest = bean._request("hello");
        helloRequest.set_return_type(orb.get_primitive_tc(TCKind.tk_long));
        helloRequest.invoke();
        return helloRequest.return_value().extract_long();
    }

    private Object createBean(ORB orb, Object beanHome) {
        Request createRequest = beanHome._request("create");
        createRequest.set_return_type(orb.get_primitive_tc(TCKind.tk_objref));
        createRequest.invoke();
        return  createRequest.return_value().extract_Object();
    }

    private Properties orbProperties(boolean enableSSL) throws Exception {
        Properties p = new Properties();
        p.setProperty("org.omg.CORBA.ORBClass", "org.jacorb.orb.ORB");
        p.setProperty("org.omg.CORBA.ORBSingletonClass", "org.jacorb.orb.ORBSingleton");
        p.setProperty("org.omg.PortableInterceptor.ORBInitializerClass", "org.jacorb.orb_positiveTest.standardInterceptors.IORInterceptorInitializer");
        if (enableSSL) {
            p.setProperty("jacorb.security.support_ssl", "on");
            p.setProperty("jacorb.ssl.socket_factory", "org.jacorb.security.ssl.sun_jsse.SSLSocketFactory");
            p.setProperty("jacorb.ssl.server_socket_factory", "org.jacorb.security.ssl.sun_jsse.SSLServerSocketFactory");
            p.setProperty("jacorb.security.keystore_password", "password");
            final URL resource = Thread.currentThread().getContextClassLoader().getResource("iiop-test.keystore");
            p.setProperty("jacorb.security.keystore", new File(resource.toURI()).getAbsolutePath());
            p.setProperty("jacorb.security.jsse.trustees_from_ks", "on");
        } else {
            p.setProperty("jacorb.security.support_ssl", "off");
        }
        return p;
    }

    private ClientEjb client() throws NamingException {
        final InitialContext context = new InitialContext();
        return (ClientEjb) context.lookup("java:module/" + ClientEjb.class.getSimpleName());
    }
}
