package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jboss.dmr.ModelNode;
import org.jboss.eap.util.xp.patch.stream.manager.server.wrapper.ServerWrapperProxy;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class RemoveUtils {
    private final ServerWrapperProxy serverWrapperProxy;
    private final Path jbossHome;

    public RemoveUtils(ServerWrapperProxy serverWrapperProxy, Path jbossHome) {
        this.serverWrapperProxy = serverWrapperProxy;
        this.jbossHome = jbossHome;
    }

    void rollbackPatchStreamPatches(List<FileSet> removeFileSets) throws Exception {
        System.out.println(ManagerLogger.LOGGER.checkingForPatchesToRollback());

        int numberPatches = 0;
        for (FileSet fileSet : removeFileSets) {
            ManagerManifestConfig.XPConfig xpConfig = fileSet.getXpConfig();
            for (String streamName : ManagerManifestConfig.INSTANCE.getXpConfig(xpConfig.getKey()).getPatchStreamNames()) {
                // TODO if these have inter-dependencies we will need to do some TBD checks. For now it is fine since there is only one
                numberPatches += rollbackPatchesForStream(serverWrapperProxy, streamName);
            }
        }
        if (numberPatches == 0) {
            System.out.println(ManagerLogger.LOGGER.noPatchesToRemove());
        }
    }

    private static int rollbackPatchesForStream(ServerWrapperProxy serverWrapperProxy, String streamName) throws Exception {
        int numberPatches = 0;
        while (true) {
            try {
                serverWrapperProxy.start();

                ModelNode idNode = serverWrapperProxy.execute(
                        new OperationBuilder("read-attribute")
                                .param("name", "cumulative-patch-id")
                                .addr("core-service", "patching")
                                .addr("patch-stream", streamName)
                                .build());

                String id = idNode.asString();

                if (id.equals("base")) {
                    break;
                }

                System.out.println(ManagerLogger.LOGGER.rollingBackPatch(id, streamName));
                serverWrapperProxy.execute(
                        new OperationBuilder("rollback")
                                .param("patch-id", id)
                                .param("reset-configuration", "false")
                                .addr("core-service", "patching")
                                .addr("patch-stream", streamName)
                                .build());
                numberPatches++;
                System.out.println(ManagerLogger.LOGGER.rolledBackPatchSuccessfully());
            } catch (Exception e) {
                System.out.println(ManagerLogger.LOGGER.errorRollingBackPatch());
                break;
            } finally {
                serverWrapperProxy.stop();
            }
        }
        return numberPatches;
    }

    void removeFiles(List<FileSet> removeFileSets) throws Exception {
        removeFiles(removeFileSets, false);
    }

    void removeFiles(List<FileSet> removeFileSets, boolean removeOnlyBackupFiles) throws Exception {
        for (FileSet fileSet : removeFileSets) {
            removeFiles(fileSet, removeOnlyBackupFiles);
        }
    }

    void removeFiles(FileSet fileSet) throws Exception {
        removeFiles(fileSet, false);
    }

    void removeFiles(FileSet fileSet, boolean removeOnlyBackupFiles) throws Exception {
        remove(fileSet.getLayersConf());
        for (Path p : fileSet.getModuleLayerDirs()) {
            remove(p);
        }
        for (Path p : fileSet.getPatchStreamFiles()) {
            remove(p);
        }
        if (!removeOnlyBackupFiles) {
            ManagerManifestConfig.XPConfig xpConfig = fileSet.getXpConfig();
            List<String> moduleLayers = xpConfig.getModuleLayerNames();
            List<String> patchNamePatterns = xpConfig.getPatchNamePatterns();
            for (String layer : moduleLayers) {
                Path p = jbossHome.resolve(".installation/layers/" + layer);
                remove(p);
            }
            for (String namePattern : patchNamePatterns) {
                Pattern pattern = Pattern.compile(namePattern);
                Path p = jbossHome.resolve(".installation/patches");
                if (Files.exists(p)) {
                    try (Stream<Path> stream = Files.list(p)) {
                        List<Path> paths = stream.collect(Collectors.toList());
                        for (Path path : paths) {
                            if (pattern.matcher(path.getFileName().toString()).matches()) {
                                remove(path);
                            }
                        }
                    }
                }
            }
        }
    }

    static void remove(Path path) {
        if (path == null) {
            return;
        }
        if (Files.exists(path)) {
            try {
                if (Files.isDirectory(path)) {
                    Files.walkFileTree(path, new FileVisitors.DeleteDirectory());
                } else {
                    Files.delete(path);
                }
            } catch (IOException e) {
                System.err.println(ManagerLogger.LOGGER.errorDeletingStateWillBeInconsistent(path));
                e.printStackTrace();
            }
        }
    }
}
