package org.jboss.eap.util.xp.patch.stream.manager;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.eap.util.xp.patch.stream.manager.ManagerManifestConfig.XPConfig;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ManagerStatus {
    private final Map<ManagerState, State> states;
    private final ManagerState managerState;
    private final State currentState;

    private ManagerStatus(Map<ManagerState, State> states, ManagerState managerState) {
        this.states = Collections.unmodifiableMap(states);
        this.managerState = managerState;
        this.currentState = states.get(managerState);
    }

    ManagerState getManagerState() {
        return managerState;
    }

    boolean hasSeveralXPPatchStreams() {
        return currentState.hasSeveralXPPatchStreams();
    }

    static ManagerStatus determineStatus(Path jbossHome, Path modulesDir) throws Exception {
        ManagerManifestConfig config = ManagerManifestConfig.INSTANCE;

        Map<String, FileSet> fileSets = new LinkedHashMap<>();
        Set<FileSet> installedSets = new LinkedHashSet<>();
        for (String key : config.getPatchStreamKeys()) {
            XPConfig xpConfig = config.getXpConfig(key);
            FileSet fileSet = FileSet.create(xpConfig, jbossHome, modulesDir);
            fileSets.put(fileSet.getXpConfig().getKey(), fileSet);

            if (fileSet.isInstalled()) {
                installedSets.add(fileSet);
            }
        }

        ManagerState managerState = ManagerState.CLEAN;
        if (installedSets.size() == 1) {
            FileSet installed = installedSets.iterator().next();
            if (installed.getXpConfig().getKey().equals(config.getCurrentPatchStreamKey())) {
                managerState = ManagerState.INSTALLED;
            } else {
                managerState = ManagerState.OLD_INSTALLED;
            }

        }
        if (installedSets.size() > 1) {
            managerState = ManagerState.INCONSISTENT;
        }

        List<FileSet> inconsistentSets = new ArrayList<>();

        // Check we don't have ones which only have modules
        for (FileSet fileSet : fileSets.values()) {
            if (installedSets.contains(fileSet)) {
                continue;
            }
            if (fileSet.getPatchStreamDirStatus() != FileSet.FileStatus.NONE) {
                managerState = ManagerState.INCONSISTENT;
                inconsistentSets.add(fileSet);
            }
            // No need to check module layers here at present as they will be the same
        }

        if (managerState == ManagerState.CLEAN || managerState == ManagerState.INCONSISTENT) {
            // Do a final pass and add sets which have modules but no patch stream
            for (FileSet fileSet : fileSets.values()) {
                if (fileSet.getModuleLayersStatus() != FileSet.FileStatus.NONE || fileSet.isLayersConfExists()) {
                    inconsistentSets.add(fileSet);
                    managerState = ManagerState.INCONSISTENT;
                }
            }
        }

        Map<ManagerState, State> states = new HashMap<>();
        states.put(ManagerState.INSTALLED,
                new InstalledState(Collections.singletonList(fileSets.get(config.getCurrentPatchStreamKey()))));
        states.put(ManagerState.CLEAN, new CleanState());

        if (managerState == ManagerState.OLD_INSTALLED) {
            states.put(ManagerState.OLD_INSTALLED, new OldInstalledState(new ArrayList<>(installedSets)));
        }
        if (managerState == ManagerState.INCONSISTENT) {
            List<FileSet> allSets = new ArrayList<>(installedSets);
            allSets.addAll(inconsistentSets);
            states.put(ManagerState.INCONSISTENT, new InconsistentState(allSets));
        }

        return new ManagerStatus(states, managerState);
    }

    Set<ManagerCommand> getAvailableCommands() {
        return currentState.getAvailableCommands();
    }

    public List<FileSet> getFileSet() {
        return currentState.fileSets;
    }

    public FileSet getToFileSet(ManagerState toManagerState) {
        return states.get(toManagerState).fileSets.get(0);
    }

    public FileSet getInstalledFileSet() {
        assert managerState == ManagerState.INSTALLED || managerState == ManagerState.OLD_INSTALLED;
        return states.get(managerState).fileSets.get(0);
    }

    public XPConfig getInstalledXPConfig() {
        return getInstalledFileSet().getXpConfig();
    }

    private static abstract class State {
        private final ManagerState managerState;
        protected final List<FileSet> fileSets;

        public State(ManagerState managerState, List<FileSet> fileSets) {
            this.managerState = managerState;
            this.fileSets = fileSets;
        }

        Set<ManagerCommand> getAvailableCommands() {
            return managerState.getAvailableCommands();
        }

        boolean hasSeveralXPPatchStreams() {
            return false;
        }
    }

    private static class CleanState extends State {
        CleanState() {
            super(ManagerState.CLEAN, Collections.emptyList());
        }
    }

    private static class InstalledState extends State {
        InstalledState(List<FileSet> fileSet) {
            super(ManagerState.INSTALLED, fileSet);
        }
    }

    private static class InconsistentState extends State {
        InconsistentState(List<FileSet> fileSet) {
            super(ManagerState.INCONSISTENT, fileSet);
        }

        @Override
        boolean hasSeveralXPPatchStreams() {
            return fileSets.size() > 1;
        }
    }

    private static class OldInstalledState extends State {
        public OldInstalledState(List<FileSet> fileSet) {
            super(ManagerState.OLD_INSTALLED, fileSet);
        }
    }
}
