/*
 * JBoss, Home of Professional Open Source
 * Copyright 2020, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.logging.annotations.Cause;
import org.jboss.logging.annotations.Message;
import org.jboss.logging.annotations.MessageLogger;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
@MessageLogger(projectCode = "MPXP", length = 4)
public interface ManagerLogger {

    ManagerLogger LOGGER = Logger.getMessageLogger(ManagerLogger.class, "org.jboss.eap.util.mp.xp");

    @Message("Starting JBoss EAP XP manager (%s).")
    String startMessage(String version);

    @Message("The JBoss EAP XP manager must be run from the distributed jar. It should not be unzipped!")
    IllegalStateException managerNotAJar();

    @Message("JBoss Enterprise Application Platform expansion pack (JBoss EAP XP or EAP XP) is subject to its own " +
            "separate support policy and life cycle for closer alignment with Eclipse MicroProfile specification " +
            "release cadence. JBoss EAP server instances with the EAP XP setup will be covered in their entirety " +
            "by the new EAP XP policy and life cycle. \n" +
            "By setting up EAP XP, this server will be subject to the EAP XP support and life cycle policy. " +
            "Please refer to the JBoss Enterprise Application Platform expansion pack Life Cycle page at " +
            "https://access.redhat.com/support/policy/updates/jboss_eap_xp_notes for more details.")
    String xpSupportPolicy();

    @Message("JBoss Enterprise Application Platform expansion pack (JBoss EAP XP or EAP XP) is subject to its own " +
            "separate support policy and life cycle for closer alignment with Eclipse MicroProfile specification " +
            "release cadence. \n" +
            "By removing EAP XP from this JBoss EAP server, this server will now be subject to the JBoss EAP support " +
            "and life cycle policy and no more to the EAP XP support policy. Please refer to the " +
            "JBoss Enterprise Application Platform Life Cycle page at " +
            "https://access.redhat.com/support/policy/updates/jboss_notes for more details.")
    String xpRemoveSupportPolicy();

    @Message("By using this argument you accept the following support policy change: \"%s\"")
    String usageAcceptSupportPolicyBase(String policy);

    default String usageAcceptSupportPolicySetup() {
        return usageAcceptSupportPolicyBase(xpSupportPolicy().replace("\n", ""));
    }

    default String usageAcceptSupportPolicyRemove() {
        return usageAcceptSupportPolicyBase(xpRemoveSupportPolicy().replace("\n", ""));
    }

    @Message("[INFO] Support policy change accepted by passing in the " + ManagerArgsParser.ARG_ACCEPT_SUPPORT_POLICY + " parameter.")
    String supportPolicyAcceptedWithArg();

    @Message("Do you accept the support and lifecycle policy change (yes/no)?")
    String acceptSupportPolicy();

    @Message("Invalid response. (Valid responses are yes, y, no and n)")
    String invalidSupportPolicyResponse();

    @Message(ManagerArgsParser.ARG_JBOSS_HOME + " was not specified, attempting to read it from the JBOSS_HOME environment variable.")
    String argsNoJBossHomeTryingEnvVar();

    @Message("[ERROR] " + ManagerArgsParser.ARG_JBOSS_HOME + " was not specified, and it was not set in the JBOSS_HOME environment variable either.")
    String argsNoJBossHome();

    @Message("[ERROR] Could not find the indicated server installation root folder at %s.")
    String jbossHomeNotFound(Path jbossHome);

    @Message("[ERROR] The indicated server installation root is not a directory: %s.")
    String jbossHomeIsNotADirectory(Path jbossHome);

    @Message("[ERROR] The determined modules base directory %s does not exist.")
    String determinedModulesBaseDirectoryDoesNotExist(Path modulesDir);

    @Message("[ERROR] The determined modules base directory %s does not look like a modules directory.")
    String determinedModulesBaseDirectoryLooksWeird(Path modulesDir);

    @Message("[ERROR] The %s argument was specified but was empty. Please supply a value.")
    String zeroLengthArgument(String arg);


    @Message("Usage: %s [command]%nwhere command may be one of:")
    String commandUsageHeadline(String executableName);

    @Message("Usage: %s %s [args...]%nwhere args include:")
    String argsUsageHeadline(String executableName, ManagerCommand command);

    @Message("Usage: %s %s")
    String noArgsUsageHeadline(String executableName, ManagerCommand command);

    @Message("[ERROR] Unknown command: %s")
    String unknownCommand(String command);

    @Message("Display this message and exit. The optional <command> argument allows you to view more details for a particular command.")
    String usageCommandHelp();

    @Message("Sets up a clean JBoss EAP server for the JBoss EAP XP patch stream.")
    String usageCommandSetup();

    @Message("Gets the status of the JBoss EAP XP patch stream setup in the JBoss EAP server.")
    String usageCommandStatus();

    @Message("Totally removes the JBoss EAP XP patch stream setup from the the JBoss EAP server.")
    String usageCommandRemove();

    @Message("Upgrades an old installed JBoss EAP XP patch stream to the latest in the JBoss EAP server.")
    String usageCommandUpgrade();

    @Message("Applies a JBoss EAP or JBoss EAP XP patch to the server.")
    String usageCommandPatchApply();

    @Message("The filesystem path of the root folder of the JBoss EAP server to manage the JBoss EAP expansion pack in.")
    String usageArgJBossHome();

    @Message("Filesystem path to a JBoss EAP patch to apply. If not present, no patch will be applied.")
    String usageArgBasePatch();

    @Message("Filesystem path to a JBoss EAP XP patch stream patch to apply. If not present, no patch will be applied.")
    String usageArgXpPatch();

    @Message("Filesystem path to a JBoss EAP or JBoss EAP XP patch to apply.")
    String usageArgPatch();

    @Message("The filesystem path of the directory to copy the new server configuration files to. If not specified, " +
            "it will use the standard standalone/configuration folder under the specified server root folder.")
    String usageArgJBossConfigDir();

    @Message("Copying server configuration file to %s")
    String copyinConfigFile(Path target);

    @Message("[WARN] There is already a %s server configuration file. It will be left in place, and the one added by this expansion pack is created at %s.")
    String copyingConfigFileAlreadyExists(Path target, Path renamed);

    @Message("The cumulative-patch-id in /core-service=patching:read-resource(recursive=true, include-runtime=true) was " +
            "'%s'. A value of 'base' or one matching the regular expression 'jboss-eap-7\\.3\\.\\d+\\.CP' " +
            "is expected in an untouched JBoss EAP server.")
    IllegalStateException badCumulativePatchId(String id);

    @Message("No '%s' entry found in any MANIFEST.MF.")
    IllegalStateException manifestEntryNotFound(String name);

    @Message("Starting the setup of the JBoss EAP expansion pack.")
    String startingSetup();

    @Message("Starting the upgrade of the JBoss EAP expansion pack.")
    String startingUpgrade();

    @Message("Setting up module layers and patch streams in the server.")
    String settingUpLayersAndStreams();

    @Message("Copying configs to be added to the server.")
    String copyingAddedConfigs();

    @Message("Starting the embedded server at %s.")
    String startingEmbeddedServer(Path jbossHome);

    @Message("Checking the server cumulative patch id.")
    String checkCumulativePatchId();

    @Message("Checking the server version.")
    String checkServerVersion();

    @Message("Applying the Jboss EAP patch: %s.")
    String applyingJBossEAPPatch(Path path);

    @Message("Applying the JBoss EAP XP patch: %s.")
    String applyingExpansionPackPatch(Path path);

    @Message("The JBoss EAP expansion pack was setup successfully. You may now use the server.")
    String expansionPackPreparedSuccessfully();

    @Message("[ERROR] Argument expected for option %s.")
    String argumentExpected(String arg);

    @Message("[ERROR] Unknown argument: %s.")
    String unknownArgument(String arg);

    @Message("[ERROR] Missing required argument: %s.")
    String missingRequiredArgument(String arg);

    @Message("[ERROR] The specified configuration directory does not exist: %s.")
    String specifiedConfigurationDirDoesNotExist(String arg);

    @Message("[ERROR] The specified configuration directory is not a directory: %s.")
    String specifiedConfigurationDirIsNotADirectory(String arg);

    @Message("[ERROR] The determined configuration directory does not exist: %s.")
    String determinedConfigurationDirectoryDoesNotExist(Path path);

    @Message("[ERROR] The determined configuration directory is not a directory: %s.")
    String determinedConfigurationDirectoryIsNotADirectory(Path path);

    @Message("[ERROR] The specified patch file does not exist: %s.")
    String givenPatchDoesNotExist(String arg);

    @Message("[ERROR] The specified patch is a directory rather than a file: %s.")
    String givenPatchDoesIsADirectory(String arg);

    @Message("JBoss EAP version %s is incompatible with the JBoss EAP XP stream. To run this manager, please upgrade to " +
            "at least JBoss EAP %s. Note that we only support certain combinations of JBoss EAP and JBoss EAP XP patches. " +
            "These combinations can be found in our documentation.")
    IllegalStateException incompatibleServerVersionApplyBasePatchFirst(ServerVersion version, ServerVersion minimumBaseServerVersion);


    @Message("JBoss EAP version %s in the patch specified by the " + ManagerArgsParser.ARG_BASE_PATCH + " argument " +
            "is incompatible with the JBoss EAP XP stream. To run this manager, please upgrade to " +
            "at least JBoss EAP %s. Note that we only support certain combinations of JBoss EAP and JBoss EAP XP patches. " +
            "These combinations can be found in our documentation.")
    IllegalStateException incompatibleBasePatchVersion(ServerVersion patchVersion, ServerVersion minimumServerVersion);

    @Message("The JBoss EAP expansion pack is currently empty. To get the intended functionality, apply patches from the JBoss EAP XP patch stream.")
    String applyXpPatchForFunctionality();

    @Message("[ERROR] An error happened resetting the status of the underlying patching system.")
    String errorResettingPatchingStatus();

    @Message("Skipping server version check since --base-patch was specified.")
    String skipServerVersionCheck();

    @Message("Specified patch file does not appear to be a valid zip archive: %s.")
    IllegalStateException patchFileIsNotAnArchive(@Cause IOException e, Path patchFile);

    @Message("Patch file specified by %s does not contain a patch.xml: %s.")
    IllegalStateException noPatchXmlFoundInPatchFile(String patchArg, Path patchFile);

    @Message("Specified patch file does not contain a valid patch id in its patch.xml. The value specified for %s "
            + " appears to be wrong. The bad patch id is: %s.")
    IllegalStateException invalidPatchIdForPatch(String patchArg, String patchId);

    @Message("Specified patch file does not contain exactly one <element> child in its patch.xml. The value specified for %s "
            + " appears to be wrong.")
    IllegalStateException patchElementsNotEqualsOneInPatch(String patchArg);

    @Message("Specified patch file contains an id for the <element> child that does not match the expected pattern. " +
            "The value specified for %s appears to be wrong. The bad element id is: %s.")
    IllegalStateException badPatchElementIdInPatch(String patchArg, String id);

    @Message("Specified patch file contains an bad name <upgrade> child that does not match the expected value. " +
            "The value specified for %s appears to be wrong. The value was '%s', while '%s' was expected.")
    IllegalStateException badUpgradeElementNameInPatch(String patchArg, String layerName, String expected);

    @Message("The JBoss EAP server does not have the JBoss EAP XP patch stream setup.%n")
    String installationStatusClean();

    @Message("The JBoss EAP XP patch stream setup in the JBoss EAP server seems broken.%n")
    String installationStatusInconsistent();

    @Message("You have more than one JBoss EAP XP patch stream set up in the JBoss EAP server. You may end " +
            "up in this situation if you installed the latest XP patch stream and applied patches from an " +
            "older JBoss EAP XP patch stream. To recover from this you need to run the 'remove' command, run " +
            "'setup' again and make sure you only install JBoss EAP XP patch stream patches from the current " +
            "JBoss EAP XP patch stream.%n")
    String installationStatusInconsistentSeveralXPPatchStreams();


    @Message("The JBoss EAP XP patch stream is enabled. You may apply patches both from the JBoss EAP and the JBoss EAP XP patch streams. While enabled " +
            "the following support policy applies:%n%n\"%s\"%n%nYou are currently on JBoss EAP XP %s. %s%n")
    String installationStatusInstalled(String supportPolicy, String version, String upgradeMessage);

    @Message("%n%nYou are using an old version of JBoss EAP XP. The current version is %s, please upgrade.")
    String oldVersionInstalledPleaseUpgrade(String version);

    @Message("Available commands in this state are: %s")
    String installationStatusAvailableCommands(Collection<ManagerCommand> commands);

    @Message("[ERROR] The command '%s' is not available in the current state of the server. Run with the 'status' command to see the current status and the available commands.")
    String commandNotAvailableInCurrentContext(ManagerCommand command);

    @Message("[ERROR] An error occurred deleting %s. The state of the JBoss EAP XP patch stream installation will be inconsistent. Run with the 'status' command to fix.")
    String errorDeletingStateWillBeInconsistent(Path path);

    @Message("[ERROR] An error happened setting up the JBoss EAP XP stream, the attempted filesystem changes will be rolled back.")
    String errorSettingUpExpansionPackRollingBackAttemptedChanges();

    @Message("Successfully removed the JBoss EAP XP stream.")
    String successfullyRemovedExpansionPack();

    @Message("Successfully removed the backed up JBoss EAP XP patch stream files.")
    String successfullyRemovedBackedUpFiles();

    @Message("Checking for patches to roll back.")
    String checkingForPatchesToRollback();

    @Message("There are no patches to roll back.")
    String noPatchesToRemove();

    @Message("Rolling back patch with id '%s' on patch stream '%s'.")
    String rollingBackPatch(String patchId, String streamName);

    @Message("Rolled back patch successfully.")
    String rolledBackPatchSuccessfully();

    @Message("[WARN] An error happened rolling back the patches. Will proceed to delete the patch stream. Some license and schema files might be left behind.")
    String errorRemovingPatches();

    @Message("[WARN] Found the log manager %s expected %s. Logging output may be unpredictable.")
    String unknownLogManager(String found, String expected);

    @Message("[ERROR] Failed to configure log manager. The The JBoss EAP XP manager will continue to work, however logging may be unpredictable.")
    String failedToConfigureLogManager();

    @Message("[ERROR] An error occurred restoring backup %s to %s.")
    String errorHappenedRestoringBackup(@Cause Exception e, Path backup, Path path);

    @Message("[ERROR] An error occurred upgrading the JBoss EAP XP patch stream. Will attempt to restore the backed up files for the originally installed JBoss EAP XP patch stream.")
    String errorUpgrading(@Cause Exception e);

    @Message("[ERROR] An error occurred restoring the JBoss EAP XP patch stream files. Your installation might be inconsistent. Run the status command to check.")
    String errorRestoringFilesInUpgradeInconsistent();

    @Message("Backing up current JBoss EAP XP patch stream files.")
    String backingUpCurrentPatchStreamFiles();

    @Message("Successfully backed up current JBoss EAP XP patch stream files.")
    String successfullyBackedUpCurrentPatchStreamFiles();

    @Message("The JBoss EAP expansion pack was successfully upgraded to JBoss EAP XP %s.")
    String successfullyUpgradedExpansionPack(String version);

    @Message("Bad patch specified in the " + ManagerArgsParser.ARG_XP_PATCH + " argument. It targets the %s patch stream, while this version of the JBoss EAP XP manager only works with the following JBoss EAP XP patch streams: %s")
    IllegalStateException invalidXpPatchForManagerVersion(String patchStreamName, Set<String> patchStreams);

    @Message("Enabled patch streams and their cumulative patch ids:")
    String installedPatchStreamsAndCumulativePatchIdsHeader();

    @Message("- Patch stream: '%s'; Cumulative patch id: '%s'")
    String patchStreamAndCumulativePatchId(String patchStream, String cumulativePatchId);

    @Message("[INFO] Patch was applied successfully: %s")
    String patchAppliedSuccessFully(String patchFileName);

    @Message("You are attempting to install a patch for the '%s' JBoss EAP XP Patch Stream. However this patch stream" +
            " is not yet set up in the JBoss EAP server. Run the 'setup' command to enable the patch stream.")
    IllegalStateException patchApplyForXPPatchCanOnlyBeAppliedOncePatchStreamIsEnabled(String patchStream);

    @Message("The JBoss EAP XP patch stream in the patch '%s' does not match the currently enabled JBoss EAP XP patch stream %s")
    IllegalStateException patchApplyForWrongXPPatchStream(String pathStream, List<String> xpStreams);

    @Message("[WARN] An error occurred trying to roll back the patch. Check the server log output for details.")
    String errorRollingBackPatch();

    @Message("[INFO] Starting external process containing embedded server.")
    String startingExternalProcessContainingEmbeddedServer();

    @Message("[INFO] Connecting to parent process on port %d.")
    String connectingToParentProcessOnPort(int port);

    @Message("[INFO] Connected to parent process.")
    String connectedToParentProcess();

    @Message("[INFO] Stopping external process containing embedded server.")
    String stoppingExternalServerInExternalProcess();

    @Message("[INFO] Attempting to start external process containing embedded server.")
    String tryingToStartExternalProcessWithEmbeddedServer();

    @Message("[INFO] External process connected.")
    String externalProcessConnected();

    @Message("[ERROR] Remote process did not connect on time.")
    IllegalStateException externalProcessDidNotConnectOnTime();

    @Message("[ERROR] An error occurred in the external remote process containing the embedded server. See stacktrace for details.")
    String anErrorHappenedInTheExternalRemoteProcess();
}

