"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheTasks = void 0;
const tslib_1 = require("tslib");
const Listr = require("listr");
const che_1 = require("../api/che");
const che_api_client_1 = require("../api/che-api-client");
const che_login_manager_1 = require("../api/che-login-manager");
const kube_1 = require("../api/kube");
const openshift_1 = require("../api/openshift");
const version_1 = require("../api/version");
const constants_1 = require("../constants");
const util_1 = require("../util");
const kube_2 = require("./kube");
/**
 * Holds tasks to work with CodeReady Workspaces component.
 */
class CheTasks {
    constructor(flags) {
        this.oc = new openshift_1.OpenShiftHelper();
        this.cheSelector = 'app=codeready,component=codeready';
        this.dashboardDeploymentName = 'che-dashboard';
        this.dashboardSelector = 'app=codeready,component=codeready-dashboard';
        this.keycloakDeploymentName = 'keycloak';
        this.keycloakSelector = 'app=codeready,component=keycloak';
        this.postgresDeploymentName = 'postgres';
        this.postgresSelector = 'app=codeready,component=postgres';
        this.devfileRegistryDeploymentName = 'devfile-registry';
        this.devfileRegistrySelector = 'app=codeready,component=devfile-registry';
        this.pluginRegistryDeploymentName = 'plugin-registry';
        this.pluginRegistrySelector = 'app=codeready,component=plugin-registry';
        this.cheConsoleLinkName = 'che';
        this.kube = new kube_1.KubeHelper(flags);
        this.kubeTasks = new kube_2.KubeTasks(flags);
        this.che = new che_1.CheHelper(flags);
        this.cheAccessToken = flags['access-token'];
        this.cheNamespace = flags.chenamespace;
        this.cheDeploymentName = flags['deployment-name'];
    }
    /**
     * Returns tasks list that waits until every CodeReady Workspaces component will be started.
     *
     * Note that CodeReady Workspaces components statuses should be already set in context.
     *
     * @see che.checkIfCheIsInstalledTasks
     */
    waitDeployedChe() {
        return [
            {
                title: 'PostgreSQL pod bootstrap',
                enabled: ctx => ctx.isPostgresDeployed && !ctx.isPostgresReady,
                task: () => this.kubeTasks.podStartTasks(this.postgresSelector, this.cheNamespace)
            },
            {
                title: 'Keycloak pod bootstrap',
                enabled: ctx => ctx.isKeycloakDeployed && !ctx.isKeycloakReady,
                task: () => this.kubeTasks.podStartTasks(this.keycloakSelector, this.cheNamespace)
            },
            {
                title: 'Devfile Registry pod bootstrap',
                enabled: ctx => ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryReady,
                task: () => this.kubeTasks.podStartTasks(this.devfileRegistrySelector, this.cheNamespace)
            },
            {
                title: 'Plug-in Registry pod bootstrap',
                enabled: ctx => ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryReady,
                task: () => this.kubeTasks.podStartTasks(this.pluginRegistrySelector, this.cheNamespace)
            },
            {
                title: 'CodeReady Workspaces Dashboard pod bootstrap',
                enabled: ctx => ctx.isDashboardDeployed && !ctx.isDashboardReady,
                task: () => this.kubeTasks.podStartTasks(this.dashboardSelector, this.cheNamespace)
            },
            {
                title: 'CodeReady Workspaces Server pod bootstrap',
                enabled: ctx => !ctx.isCheReady,
                task: () => this.kubeTasks.podStartTasks(this.cheSelector, this.cheNamespace)
            },
            ...this.checkEclipseCheStatus()
        ];
    }
    /**
     * Returns list of tasks that checks if CodeReady Workspaces is already installed.
     *
     * After executing the following properties are set in context:
     * is[Component]Deployed, is[Component]Stopped, is[Component]Ready
     * where component is one the: Che, Keycloak, Postgres, PluginRegistry, DevfileRegistry
     */
    checkIfCheIsInstalledTasks(_flags, command) {
        return [
            {
                title: `Verify if CodeReady Workspaces is deployed into namespace \"${this.cheNamespace}\"`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kube.deploymentExist(this.cheDeploymentName, this.cheNamespace)) {
                        // helm chart and CodeReady Workspaces operator use a deployment
                        ctx.isCheDeployed = true;
                        ctx.isCheReady = yield this.kube.deploymentReady(this.cheDeploymentName, this.cheNamespace);
                        if (!ctx.isCheReady) {
                            ctx.isCheStopped = yield this.kube.deploymentStopped(this.cheDeploymentName, this.cheNamespace);
                        }
                        ctx.isDashboardDeployed = yield this.kube.deploymentExist(this.dashboardDeploymentName, this.cheNamespace);
                        if (ctx.isDashboardDeployed) {
                            ctx.isDashboardReady = yield this.kube.deploymentReady(this.dashboardDeploymentName, this.cheNamespace);
                            if (!ctx.isDashboardReady) {
                                ctx.isDashboardStopped = yield this.kube.deploymentStopped(this.dashboardDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isKeycloakDeployed = yield this.kube.deploymentExist(this.keycloakDeploymentName, this.cheNamespace);
                        if (ctx.isKeycloakDeployed) {
                            ctx.isKeycloakReady = yield this.kube.deploymentReady(this.keycloakDeploymentName, this.cheNamespace);
                            if (!ctx.isKeycloakReady) {
                                ctx.isKeycloakStopped = yield this.kube.deploymentStopped(this.keycloakDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isPostgresDeployed = yield this.kube.deploymentExist(this.postgresDeploymentName, this.cheNamespace);
                        if (ctx.isPostgresDeployed) {
                            ctx.isPostgresReady = yield this.kube.deploymentReady(this.postgresDeploymentName, this.cheNamespace);
                            if (!ctx.isPostgresReady) {
                                ctx.isPostgresStopped = yield this.kube.deploymentStopped(this.postgresDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isDevfileRegistryDeployed = yield this.kube.deploymentExist(this.devfileRegistryDeploymentName, this.cheNamespace);
                        if (ctx.isDevfileRegistryDeployed) {
                            ctx.isDevfileRegistryReady = yield this.kube.deploymentReady(this.devfileRegistryDeploymentName, this.cheNamespace);
                            if (!ctx.isDevfileRegistryReady) {
                                ctx.isDevfileRegistryStopped = yield this.kube.deploymentStopped(this.devfileRegistryDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isPluginRegistryDeployed = yield this.kube.deploymentExist(this.pluginRegistryDeploymentName, this.cheNamespace);
                        if (ctx.isPluginRegistryDeployed) {
                            ctx.isPluginRegistryReady = yield this.kube.deploymentReady(this.pluginRegistryDeploymentName, this.cheNamespace);
                            if (!ctx.isPluginRegistryReady) {
                                ctx.isPluginRegistryStopped = yield this.kube.deploymentStopped(this.pluginRegistryDeploymentName, this.cheNamespace);
                            }
                        }
                    }
                    if (!ctx.isCheDeployed) {
                        task.title = `${task.title}...it is not`;
                    }
                    else {
                        return new Listr([
                            {
                                enabled: () => ctx.isCheDeployed,
                                title: `Found ${ctx.isCheStopped ? 'stopped' : 'running'} CodeReady Workspaces deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isPostgresDeployed,
                                title: `Found ${ctx.isPostgresStopped ? 'stopped' : 'running'} postgres deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isKeycloakDeployed,
                                title: `Found ${ctx.isKeycloakStopped ? 'stopped' : 'running'} keycloak deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isPluginRegistryDeployed,
                                title: `Found ${ctx.isPluginRegistryStopped ? 'stopped' : 'running'} plugin registry deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isDevfileRegistryDeployed,
                                title: `Found ${ctx.isDevfileRegistryStopped ? 'stopped' : 'running'} devfile registry deployment`,
                                task: () => { }
                            }
                        ]);
                    }
                })
            },
            {
                title: 'Check CodeReady Workspaces server status',
                enabled: (ctx) => ctx.isCheDeployed && ctx.isCheReady,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    let cheURL = '';
                    try {
                        cheURL = yield this.che.cheURL(this.cheNamespace);
                        const cheApi = che_api_client_1.CheApiClient.getInstance(cheURL + '/api');
                        const status = yield cheApi.getCheServerStatus();
                        ctx.isAuthEnabled = yield cheApi.isAuthenticationEnabled();
                        const auth = ctx.isAuthEnabled ? '(auth enabled)' : '(auth disabled)';
                        task.title = `${task.title}...${status} ${auth}`;
                    }
                    catch (error) {
                        command.error(`E_CHECK_CHE_STATUS_FAIL - Failed to check CodeReady Workspaces status (URL: ${cheURL}). ${error.message}`);
                    }
                })
            }
        ];
    }
    /**
     * Returns tasks list which scale up all CodeReady Workspaces components which are deployed.
     * It requires {@link this#checkIfCheIsInstalledTasks} to be executed before.
     *
     * @see [CheTasks](#checkIfCheIsInstalledTasks)
     */
    scaleCheUpTasks() {
        return [
            {
                title: 'PostgreSQL pod bootstrap',
                enabled: ctx => ctx.isPostgresDeployed && !ctx.isPostgresReady,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.scaleDeployment(this.postgresDeploymentName, this.cheNamespace, 1);
                    return this.kubeTasks.podStartTasks(this.postgresSelector, this.cheNamespace);
                })
            },
            {
                title: 'Keycloak pod bootstrap',
                enabled: ctx => ctx.isKeycloakDeployed && !ctx.isKeycloakReady,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.scaleDeployment(this.keycloakDeploymentName, this.cheNamespace, 1);
                    return this.kubeTasks.podStartTasks(this.keycloakSelector, this.cheNamespace);
                })
            },
            {
                title: 'Devfile registry pod bootstrap',
                enabled: ctx => ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryReady,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.scaleDeployment(this.devfileRegistryDeploymentName, this.cheNamespace, 1);
                    return this.kubeTasks.podStartTasks(this.devfileRegistrySelector, this.cheNamespace);
                })
            },
            {
                title: 'Plug-in Registry pod bootstrap',
                enabled: ctx => ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryReady,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.scaleDeployment(this.pluginRegistryDeploymentName, this.cheNamespace, 1);
                    return this.kubeTasks.podStartTasks(this.pluginRegistrySelector, this.cheNamespace);
                })
            },
            {
                title: 'CodeReady Workspaces Dashboard pod bootstrap',
                enabled: ctx => ctx.isDashboardDeployed && !ctx.isDashboardReady,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.scaleDeployment(this.dashboardDeploymentName, this.cheNamespace, 1);
                    return this.kubeTasks.podStartTasks(this.dashboardSelector, this.cheNamespace);
                })
            },
            {
                title: 'CodeReady Workspaces Server pod bootstrap',
                enabled: ctx => ctx.isCheDeployed && !ctx.isCheReady,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.scaleDeployment(this.cheDeploymentName, this.cheNamespace, 1);
                    return this.kubeTasks.podStartTasks(this.cheSelector, this.cheNamespace);
                })
            },
            ...this.checkEclipseCheStatus()
        ];
    }
    /**
     * Returns tasks list which scale down all CodeReady Workspaces components which are deployed.
     * It requires {@link this#checkIfCheIsInstalledTasks} to be executed before.
     *
     * @see [CheTasks](#checkIfCheIsInstalledTasks)
     */
    scaleCheDownTasks(command) {
        return [{
                title: 'Stop CodeReady Workspaces Server and wait until it\'s ready to shutdown',
                enabled: (ctx) => !ctx.isCheStopped,
                task: (task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        const cheURL = yield this.che.cheURL(this.cheNamespace);
                        const cheApi = che_api_client_1.CheApiClient.getInstance(cheURL + '/api');
                        let cheAccessToken = this.cheAccessToken;
                        if (!cheAccessToken && (yield cheApi.isAuthenticationEnabled())) {
                            const loginManager = yield che_login_manager_1.CheServerLoginManager.getInstance();
                            cheAccessToken = yield loginManager.getNewAccessToken();
                        }
                        yield cheApi.startCheServerShutdown(cheAccessToken);
                        yield cheApi.waitUntilCheServerReadyToShutdown();
                        task.title = `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SHUTDOWN_CHE_SERVER_FAIL - Failed to shutdown CodeReady Workspaces server. ${error.message}`);
                    }
                })
            },
            {
                title: `Scale \"${this.cheDeploymentName}\" deployment to zero`,
                enabled: (ctx) => !ctx.isCheStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.cheDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"dashboard\" deployment to zero',
                enabled: (ctx) => ctx.isDashboardDeployed && !ctx.isDashboardStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.dashboardDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale dashboard deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"keycloak\" deployment to zero',
                enabled: (ctx) => ctx.isKeycloakDeployed && !ctx.isKeycloakStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.keycloakDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale keycloak deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"postgres\" deployment to zero',
                enabled: (ctx) => ctx.isPostgresDeployed && !ctx.isPostgresStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.postgresDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale postgres deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"devfile registry\" deployment to zero',
                enabled: (ctx) => ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.devfileRegistryDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale devfile-registry deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"plugin registry\" deployment to zero',
                enabled: (ctx) => ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.pluginRegistryDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale plugin-registry deployment. ${error.message}`);
                    }
                })
            }];
    }
    /**
     * Returns tasks which remove all CodeReady Workspaces related resources.
     */
    deleteTasks(flags) {
        return [
            {
                title: 'Delete all deployments',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteAllDeployments(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all services',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteAllServices(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all ingresses',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteAllIngresses(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all routes',
                enabled: (ctx) => ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.oc.deleteAllRoutes(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete configmaps for CodeReady Workspaces server and operator',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteConfigMap('che', flags.chenamespace);
                    yield this.kube.deleteConfigMap('codeready-operator', flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete rolebindings che, che-workspace-exec and che-workspace-view',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteRoleBinding('che', flags.chenamespace);
                    yield this.kube.deleteRoleBinding('codeready-operator', flags.chenamespace);
                    yield this.kube.deleteRoleBinding('che-workspace-exec', flags.chenamespace);
                    yield this.kube.deleteRoleBinding('che-workspace-view', flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete service accounts che, che-workspace',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteServiceAccount('che', flags.chenamespace);
                    yield this.kube.deleteServiceAccount('che-workspace', flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete PVCs',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deletePersistentVolumeClaim('postgres-data', flags.chenamespace);
                    yield this.kube.deletePersistentVolumeClaim('che-data-volume', flags.chenamespace);
                    yield this.kube.deletePersistentVolumeClaim('keycloak-data', flags.chenamespace);
                    yield this.kube.deletePersistentVolumeClaim('keycloak-log', flags.chenamespace);
                    task.title = `${task.title}...OK`;
                })
            },
            {
                title: `Delete consoleLink ${this.cheConsoleLinkName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const checlusters = yield this.kube.getAllCheClusters();
                    // Delete the consoleLink only in case if there no more checluster installed
                    if (checlusters.length === 0) {
                        yield this.kube.deleteConsoleLink(this.cheConsoleLinkName);
                    }
                    task.title = `${task.title}...OK`;
                })
            }
        ];
    }
    /**
     * Returns tasks which wait until pods are deleted.
     */
    waitPodsDeletedTasks() {
        return [
            {
                title: 'Wait until CodeReady Workspaces Server pod is deleted',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.cheSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until CodeReady Workspaces Dashboard pod is deleted',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.dashboardSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Keycloak pod is deleted',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.keycloakSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until PostgreSQL pod is deleted',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.postgresSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Devfile Registry pod is deleted',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.devfileRegistrySelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Plug-in Registry pod is deleted',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.pluginRegistrySelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            }
        ];
    }
    deleteNamespace(flags) {
        return [{
                title: `Delete namespace ${flags.chenamespace}`,
                task: (task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const namespaceExist = yield this.kube.getNamespace(flags.chenamespace);
                    if (namespaceExist) {
                        yield this.kube.deleteNamespace(flags.chenamespace);
                    }
                    task.title = `${task.title}...OK`;
                })
            }];
    }
    verifyCheNamespaceExistsTask(flags, command) {
        return [{
                title: `Verify if namespace '${flags.chenamespace}' exists`,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield this.kube.getNamespace(flags.chenamespace))) {
                        command.error(`E_BAD_NS - Namespace does not exist.\nThe Kubernetes Namespace "${flags.chenamespace}" doesn't exist.`, { code: 'EBADNS' });
                    }
                })
            }];
    }
    /**
     * Verifies if workspace running and puts #V1Pod into a context.
     */
    verifyWorkspaceRunTask(flags, command) {
        return [{
                title: 'Verify if the workspaces is running',
                task: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.pod = yield this.che.getWorkspacePodName(flags.chenamespace, flags.workspace).catch(e => command.error(e.message));
                })
            }];
    }
    /**
     * Return tasks to collect CodeReady Workspaces logs.
     */
    serverLogsTasks(flags, follow) {
        return [
            {
                title: `${follow ? 'Start following' : 'Read'} Operator logs`,
                skip: () => flags.installer === 'helm',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, constants_1.CHE_OPERATOR_SELECTOR, ctx.directory, follow);
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} CodeReady Workspaces Server logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.cheSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} PostgreSQL logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.postgresSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Keycloak logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.keycloakSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Plug-in Registry logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.pluginRegistrySelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Devfile Registry logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.devfileRegistrySelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} CodeReady Workspaces Dashboard logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.dashboardSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} namespace events`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readNamespaceEvents(flags.chenamespace, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            }
        ];
    }
    debugTask(flags) {
        return [
            {
                title: 'Find CodeReady Workspaces Server pod',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const chePods = yield this.kube.listNamespacedPod(flags.chenamespace, undefined, this.cheSelector);
                    if (chePods.items.length === 0) {
                        throw new Error(`CodeReady Workspaces server pod not found in the namespace '${flags.chenamespace}'`);
                    }
                    ctx.podName = chePods.items[0].metadata.name;
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: 'Check if debug mode is enabled',
                task: (task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const configMap = yield this.kube.getConfigMap('che', flags.chenamespace);
                    if (!configMap || configMap.data.CHE_DEBUG_SERVER !== 'true') {
                        throw new Error('CodeReady Workspaces server should be redeployed with \'--debug\' flag');
                    }
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: `Forward port '${flags['debug-port']}'`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.portForward(ctx.podName, flags.chenamespace, flags['debug-port']);
                    task.title = `${task.title}...done`;
                })
            }
        ];
    }
    preparePostInstallationOutput(flags) {
        return [
            {
                title: 'Prepare post installation output',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const messages = [];
                    const version = yield version_1.VersionHelper.getCheVersion(flags);
                    messages.push(`CodeReady Workspaces ${version.trim()} has been successfully deployed.`);
                    messages.push(`Documentation             : ${constants_1.DOC_LINK}`);
                    if (constants_1.DOC_LINK_RELEASE_NOTES) {
                        messages.push(`Release Notes           : ${constants_1.DOC_LINK_RELEASE_NOTES}`);
                    }
                    messages.push(constants_1.OUTPUT_SEPARATOR);
                    const cheUrl = yield this.che.cheURL(flags.chenamespace);
                    messages.push(`Users Dashboard           : ${cheUrl}`);
                    const cr = yield this.kube.getCheCluster(flags.chenamespace);
                    if (ctx.isOpenShift && cr && cr.spec && cr.spec.auth && cr.spec.auth.openShiftoAuth) {
                        if (cr.status && cr.status.openShiftOAuthUserCredentialsSecret) {
                            let user = '';
                            let password = '';
                            // read secret from the `openshift-config` namespace
                            let credentialsSecret = yield this.kube.getSecret(cr.status.openShiftOAuthUserCredentialsSecret, 'openshift-config');
                            if (credentialsSecret) {
                                user = util_1.base64Decode(credentialsSecret.data.user);
                                password = util_1.base64Decode(credentialsSecret.data.password);
                            }
                            else {
                                // read legacy secret from the `flags.chenamespace` namespace
                                credentialsSecret = yield this.kube.getSecret(cr.status.openShiftOAuthUserCredentialsSecret, flags.chenamespace);
                                if (credentialsSecret) {
                                    user = util_1.base64Decode(credentialsSecret.data.user);
                                    password = util_1.base64Decode(credentialsSecret.data.password);
                                }
                            }
                            if (user && password) {
                                messages.push(`HTPasswd user credentials : "${user}:${password}".`);
                            }
                        }
                    }
                    else {
                        messages.push('Admin user login          : "admin:admin". NOTE: must change after first login.');
                    }
                    messages.push(constants_1.OUTPUT_SEPARATOR);
                    const cheConfigMap = yield this.kube.getConfigMap('che', flags.chenamespace);
                    if (cheConfigMap && cheConfigMap.data) {
                        if (cheConfigMap.data.CHE_WORKSPACE_PLUGIN__REGISTRY__URL) {
                            messages.push(`Plug-in Registry          : ${cheConfigMap.data.CHE_WORKSPACE_PLUGIN__REGISTRY__URL}`);
                        }
                        if (cheConfigMap.data.CHE_WORKSPACE_DEVFILE__REGISTRY__URL) {
                            messages.push(`Devfile Registry          : ${cheConfigMap.data.CHE_WORKSPACE_DEVFILE__REGISTRY__URL}`);
                        }
                        messages.push(constants_1.OUTPUT_SEPARATOR);
                        if (cheConfigMap.data.CHE_KEYCLOAK_AUTH__SERVER__URL) {
                            messages.push(`Identity Provider URL     : ${cheConfigMap.data.CHE_KEYCLOAK_AUTH__SERVER__URL}`);
                        }
                        if (ctx.identityProviderUsername && ctx.identityProviderPassword) {
                            messages.push(`Identity Provider login   : "${ctx.identityProviderUsername}:${ctx.identityProviderPassword}".`);
                        }
                        messages.push(constants_1.OUTPUT_SEPARATOR);
                    }
                    ctx.highlightedMessages = messages.concat(ctx.highlightedMessages);
                    task.title = `${task.title}...done`;
                })
            }
        ];
    }
    checkEclipseCheStatus() {
        return [
            {
                title: 'CodeReady Workspaces status check',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const cheApi = che_api_client_1.CheApiClient.getInstance(ctx.cheURL + '/api');
                    task.title = `${task.title}...done`;
                    return cheApi.isCheServerReady();
                })
            }
        ];
    }
}
exports.CheTasks = CheTasks;
//# sourceMappingURL=che.js.map