"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const execa = require("execa");
const che_api_client_1 = require("../../api/che-api-client");
const che_login_manager_1 = require("../../api/che-login-manager");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const REFRESH_TOKEN_KEY = 'refresh-token';
const PASSWORD_KEY = 'password';
class Login extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { args, flags } = this.parse(Login);
            yield context_1.ChectlContext.init(flags, this);
            // Not recommended to track user and password in telemetry
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Login.id, flags });
            const loginManager = yield che_login_manager_1.CheServerLoginManager.getInstance();
            let cheApiClient;
            let cheApiEndpoint = args[common_flags_1.CHE_API_ENDPOINT_KEY];
            if (!cheApiEndpoint) {
                cheApiEndpoint = yield che_login_manager_1.getCheApiEndpoint(flags);
                cli_ux_1.cli.info(`Using ${cheApiEndpoint} server API URL to log in`);
                cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            }
            else {
                cheApiEndpoint = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(cheApiEndpoint);
                cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
                try {
                    yield cheApiClient.checkCheApiEndpointUrl();
                }
                catch (error) {
                    // Wrong API URL, try to guess, maybe base url is provided
                    if (!cheApiEndpoint.endsWith('api')) {
                        cheApiEndpoint += '/api';
                        cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
                        yield cheApiClient.checkCheApiEndpointUrl();
                    }
                    else {
                        throw error;
                    }
                }
            }
            if (!(yield cheApiClient.isAuthenticationEnabled())) {
                cli_ux_1.cli.info(`Authentication is not supported on the server: "${cheApiEndpoint}"`);
                return;
            }
            // Try to login user
            const refreshToken = flags[REFRESH_TOKEN_KEY];
            const username = flags[common_flags_1.USERNAME_KEY];
            let loginData;
            if (refreshToken) {
                loginData = { refreshToken, expires: Date.now() / 1000 + 60 };
            }
            else if (username) {
                let password = flags[PASSWORD_KEY];
                if (!password) {
                    // Password wasn't provided, ask user to input it
                    password = yield cli_ux_1.cli.prompt(`Password for ${flags.username} on ${cheApiEndpoint}`, { type: 'hide' });
                    if (!password) {
                        throw new Error('Password is required');
                    }
                }
                loginData = { username, password };
            }
            else {
                const kube = new kube_1.KubeHelper(flags);
                // User is logged into cluster with oc or kubectl
                // Try to retrieve oc user token
                if (yield kube.isOpenShift()) {
                    let ocUserToken;
                    const getUserTokenArgs = ['whoami', '--show-token'];
                    try {
                        ocUserToken = (yield execa(util_1.OPENSHIFT_CLI, getUserTokenArgs, { timeout: 10000 })).stdout;
                    }
                    catch (_a) {
                        // Che is running on a Kubernetes cluster
                        throw new Error(`No credentials provided. Please provide "--${REFRESH_TOKEN_KEY}" or "--${common_flags_1.USERNAME_KEY}" parameter`);
                    }
                    const subjectIssuer = (yield kube.isOpenShift4()) ? 'openshift-v4' : 'openshift-v3';
                    loginData = { subjectToken: ocUserToken, subjectIssuer };
                }
                else {
                    const username = yield cli_ux_1.cli.prompt(`Username on ${cheApiEndpoint}`);
                    if (!username) {
                        throw new Error('Username is required');
                    }
                    const password = yield cli_ux_1.cli.prompt(`Password for ${username} on ${cheApiEndpoint}`, { type: 'hide' });
                    if (!password) {
                        throw new Error('Password is required');
                    }
                    loginData = { username, password };
                }
            }
            if (!loginData) {
                throw new Error('Login data is required. Please provide token or username and password.');
            }
            try {
                const username = yield loginManager.setLoginContext(cheApiEndpoint, loginData);
                cli_ux_1.cli.info(`Successfully logged into ${cheApiEndpoint} as ${username}`);
            }
            catch (err) {
                this.error(util_1.getCommandErrorMessage(err));
            }
        });
    }
}
exports.default = Login;
Login.description = 'Log in to CodeReady Workspaces server';
Login.args = [
    {
        name: common_flags_1.CHE_API_ENDPOINT_KEY,
        description: 'CodeReady Workspaces server API endpoint',
        env: 'CHE_API_ENDPOINT',
        required: false // In case of login via oc token with admin rights
    }
];
Login.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    [REFRESH_TOKEN_KEY]: flags_1.string({
        char: 't',
        description: 'Keycloak refresh token',
        env: 'CHE_KEYCLOAK_REFRESH_TOKEN',
        required: false,
        exclusive: [common_flags_1.USERNAME_KEY, PASSWORD_KEY]
    }),
    [common_flags_1.USERNAME_KEY]: common_flags_1.username,
    [PASSWORD_KEY]: flags_1.string({
        char: 'p',
        description: 'CodeReady Workspaces user password',
        env: 'CHE_USER_PASSWORD',
        required: false,
        exclusive: [REFRESH_TOKEN_KEY]
    }),
    telemetry: common_flags_1.CHE_TELEMETRY
};
Login.examples = [
    '# Log in with username and password (when OpenShift OAuth is not enabled):\n' +
        'crwctl auth:login https://che-che.apps-crc.testing/api -u username -p password',
    '\n\n# Log in with username and password (password will be asked interactively):\n' +
        'crwctl auth:login che-che.apps-crc.testing -u username',
    '\n\n# Log in with token (when OpenShift OAuth is enabled):\n' +
        'crwctl auth:login che.openshift.io -t token',
    '\n\n# Log in with oc token (when logged into an OpenShift cluster with oc and OpenShift OAuth is enabled):\n' +
        'crwctl auth:login che.my.server.net',
];
//# sourceMappingURL=login.js.map