"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KubeTasks = void 0;
const tslib_1 = require("tslib");
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
const cli_ux_1 = require("cli-ux");
const Listr = require("listr");
const kube_1 = require("../api/kube");
class KubeTasks {
    constructor(flags) {
        this.interval = 500;
        this.kubeHelper = new kube_1.KubeHelper(flags);
    }
    podStartTasks(selector, namespace) {
        return new Listr([
            {
                title: 'Scheduling',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const iterations = this.kubeHelper.podWaitTimeout / this.interval;
                    for (let i = 1; i <= iterations; i++) {
                        // check cheCluster status
                        const cheClusterFailState = yield this.getCheClusterFailState(namespace);
                        if (cheClusterFailState) {
                            task.title = `${task.title}...failed`;
                            throw new Error(`CodeReady Workspaces operator failed, reason: ${cheClusterFailState.reason}, message: ${cheClusterFailState.message}. Consider increasing error recheck timeout with --k8spoderrorrechecktimeout flag.`);
                        }
                        // check 'PodScheduled' condition
                        const podFailState = yield this.getPodFailState(namespace, selector, 'PodScheduled');
                        if (podFailState) {
                            // for instance we need some time for pvc provisioning...
                            yield cli_ux_1.cli.wait(this.kubeHelper.podErrorRecheckTimeout);
                            const podFailState = yield this.getPodFailState(namespace, selector, 'PodScheduled');
                            if (podFailState) {
                                task.title = `${task.title}...failed`;
                                throw new Error(`Failed to schedule a pod, reason: ${podFailState.reason}, message: ${podFailState.message}. Consider increasing error recheck timeout with --k8spoderrorrechecktimeout flag.`);
                            }
                        }
                        const allScheduled = yield this.isPodConditionStatusPassed(namespace, selector, 'PodScheduled');
                        if (allScheduled) {
                            task.title = `${task.title}...done`;
                            return;
                        }
                        yield cli_ux_1.cli.wait(this.interval);
                    }
                    throw new Error(`Failed to schedule a pod: ${yield this.getTimeOutErrorMessage(namespace, selector)}`);
                }),
            },
            {
                title: 'Downloading images',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const iterations = this.kubeHelper.podDownloadImageTimeout / this.interval;
                    for (let i = 1; i <= iterations; i++) {
                        const failedState = yield this.getContainerFailState(namespace, selector, 'Pending');
                        if (failedState) {
                            yield cli_ux_1.cli.wait(this.kubeHelper.podErrorRecheckTimeout);
                            const failedState = yield this.getContainerFailState(namespace, selector, 'Pending');
                            if (failedState) {
                                task.title = `${task.title}...failed`;
                                throw new Error(`Failed to download image, reason: ${failedState.reason}, message: ${failedState.message}.`);
                            }
                        }
                        const pods = yield this.kubeHelper.getPodListByLabel(namespace, selector);
                        const allRunning = !pods.some(value => !value.status || value.status.phase !== 'Running');
                        if (pods.length && allRunning) {
                            task.title = `${task.title}...done`;
                            return;
                        }
                        yield cli_ux_1.cli.wait(this.interval);
                    }
                    throw new Error(`Failed to download image: ${yield this.getTimeOutErrorMessage(namespace, selector)}`);
                }),
            },
            {
                title: 'Starting',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const iterations = this.kubeHelper.podReadyTimeout / this.interval;
                    for (let i = 1; i <= iterations; i++) {
                        // check cheCluster status
                        const cheClusterFailState = yield this.getCheClusterFailState(namespace);
                        if (cheClusterFailState) {
                            task.title = `${task.title}...failed`;
                            throw new Error(`CodeReady Workspaces operator failed, reason: ${cheClusterFailState.reason}, message: ${cheClusterFailState.message}. Consider increasing error recheck timeout with --k8spoderrorrechecktimeout flag.`);
                        }
                        const failedState = yield this.getContainerFailState(namespace, selector, 'Running');
                        if (failedState) {
                            yield cli_ux_1.cli.wait(this.kubeHelper.podErrorRecheckTimeout);
                            const failedState = yield this.getContainerFailState(namespace, selector, 'Running');
                            if (failedState) {
                                task.title = `${task.title}...failed`;
                                throw new Error(`Failed to start a pod, reason: ${failedState.reason}, message: ${failedState.message}`);
                            }
                        }
                        const terminatedState = yield this.kubeHelper.getPodLastTerminatedState(namespace, selector);
                        if (terminatedState) {
                            task.title = `${task.title}...failed`;
                            let errorMsg = `Failed to start a pod, reason: ${terminatedState.reason}`;
                            terminatedState.message && (errorMsg += `, message: ${terminatedState.message}`);
                            terminatedState.exitCode && (errorMsg += `, exitCode: ${terminatedState.exitCode}`);
                            terminatedState.signal && (errorMsg += `, signal: ${terminatedState.signal}`);
                            throw new Error(errorMsg);
                        }
                        const allStarted = yield this.isPodConditionStatusPassed(namespace, selector, 'Ready');
                        if (allStarted) {
                            task.title = `${task.title}...done`;
                            return;
                        }
                        yield cli_ux_1.cli.wait(this.interval);
                    }
                    throw new Error(`Failed to start a pod: ${yield this.getTimeOutErrorMessage(namespace, selector)}`);
                }),
            },
        ]);
    }
    getPodFailState(namespace, selector, conditionType) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const status = yield this.kubeHelper.getPodCondition(namespace, selector, conditionType);
            return status.find(s => s.status === 'False' && s.message && s.reason);
        });
    }
    isPodConditionStatusPassed(namespace, selector, conditionType) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const status = yield this.kubeHelper.getPodCondition(namespace, selector, conditionType);
            const allScheduled = !status.some(s => s.status !== 'True');
            return Boolean(status.length) && allScheduled;
        });
    }
    /**
     * Checks if there is any reason for a given pod state and returns message if so.
     */
    getContainerFailState(namespace, selector, state) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const waitingState = yield this.kubeHelper.getPodWaitingState(namespace, selector, state);
            if (waitingState && waitingState.reason && waitingState.message) {
                return waitingState;
            }
        });
    }
    /**
     * Returns extended timeout error message explaining a failure.
     */
    getTimeOutErrorMessage(namespace, selector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pods = yield this.kubeHelper.getPodListByLabel(namespace, selector);
            if (!pods.length) {
                throw new Error(`Timeout: there are no pods in the namespace: ${namespace}, selector: ${selector}. Check CodeReady Workspaces logs for details. Consider increasing error recheck timeout with --k8spoderrorrechecktimeout flag.`);
            }
            let errorMessage = 'Timeout:';
            for (const pod of pods) {
                errorMessage += `\nPod: ${pod.metadata.name}`;
                if (pod.status) {
                    if (pod.status.containerStatuses) {
                        errorMessage += `\n\t\tstatus: ${JSON.stringify(pod.status.containerStatuses, undefined, '  ')}`;
                    }
                    if (pod.status.conditions) {
                        errorMessage += `\n\t\tconditions: ${JSON.stringify(pod.status.conditions, undefined, '  ')}`;
                    }
                }
                else {
                    errorMessage += ', status not found.';
                }
            }
            return errorMessage;
        });
    }
    getCheClusterFailState(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCluster = yield this.kubeHelper.getCheCluster(namespace);
            if (cheCluster && cheCluster.status && cheCluster.status.reason && cheCluster.status.message) {
                return cheCluster.status;
            }
        });
    }
}
exports.KubeTasks = KubeTasks;
//# sourceMappingURL=kube.js.map