"use strict";
/**
 * Copyright (c) 2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const Listr = require("listr");
const constants_1 = require("../../constants");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const backup_restore_1 = require("../../api/backup-restore");
const cli_ux_1 = require("cli-ux");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
const backup_1 = require("./backup");
class Restore extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Restore);
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Restore.id, flags });
            const tasks = new Listr([], ctx.listrOptions);
            const apiTasks = new api_1.ApiTasks();
            tasks.add(apiTasks.testApiTasks(flags));
            tasks.add(this.getRestoreTasks(flags));
            try {
                yield tasks.run(ctx);
            }
            catch (err) {
                this.error(util_1.wrapCommandError(err));
            }
            cli_ux_1.cli.info(util_1.getCommandSuccessMessage());
            util_1.notifyCommandCompletedSuccessfully();
        });
    }
    getRestoreTasks(flags) {
        return [
            {
                title: 'Scheduling restore...',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const backupServerConfig = backup_1.getBackupServerConfiguration(flags);
                    yield backup_restore_1.requestRestore(flags.chenamespace, backupServerConfig, flags['snapshot-id']);
                    task.title = `${task.title}OK`;
                }),
            },
            {
                title: 'Waiting until restore process finishes...',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const kube = new kube_1.KubeHelper(flags);
                    let restoreStatus = {};
                    do {
                        yield cli_ux_1.cli.wait(1000);
                        const restoreCr = yield kube.getCustomResource(flags.chenamespace, constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_CLUSTER_RESTORE_KIND_PLURAL);
                        if (!restoreCr.status) {
                            continue;
                        }
                        restoreStatus = restoreCr.status;
                        if (restoreStatus.stage) {
                            task.title = `Waiting until restore process finishes: ${restoreStatus.stage}`;
                        }
                    } while (restoreStatus.state === 'InProgress');
                    if (restoreStatus.state === 'Failed') {
                        throw new Error(`Failed to restore installation: ${restoreStatus.message}`);
                    }
                    task.title = 'Waiting until restore process finishes...OK';
                }),
            },
        ];
    }
}
exports.default = Restore;
Restore.description = 'Restore CodeReady Workspaces installation';
Restore.examples = [
    '# Reuse existing backup configuration:\n' +
        'crwctl server:restore',
    '# Restore from specific backup snapshot using previos backup configuration:\n' +
        'crwctl server:restore -s 585421f3',
    '# Create and use configuration for REST backup server:\n' +
        'crwctl server:resotre -r rest:http://my-sert-server.net:4000/che-backup -p repopassword',
    '# Create and use configuration for AWS S3 (and API compatible) backup server (bucket should be precreated):\n' +
        'crwctl server:backup -r s3:s3.amazonaws.com/bucketche -p repopassword',
    '# Create and use configuration for SFTP backup server:\n' +
        'crwctl server:backup -r=sftp:user@my-server.net:/srv/sftp/che-data -p repopassword',
];
Restore.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    [backup_1.BACKUP_REPOSITORY_URL_KEY]: backup_1.backupRepositoryUrl,
    [backup_1.BACKUP_REPOSITORY_PASSWORD_KEY]: backup_1.backupRepositoryPassword,
    [backup_1.BACKUP_REST_SERVER_USERNAME_KEY]: backup_1.backupRestServerUsername,
    [backup_1.BACKUP_REST_SERVER_PASSWORD_KEY]: backup_1.backupRestServerPassword,
    [backup_1.SSH_KEY_KEY]: backup_1.sshKey,
    [backup_1.SSH_KEY_FILE_KEY]: backup_1.sshKeyFile,
    [backup_1.AWS_ACCESS_KEY_ID_KEY]: backup_1.awsAccessKeyId,
    [backup_1.AWS_SECRET_ACCESS_KEY_KEY]: backup_1.awsSecretAccessKey,
    [backup_1.BACKUP_SERVER_CONFIG_CR_NAME_KEY]: backup_1.backupServerConfigName,
    'snapshot-id': flags_1.string({
        char: 's',
        description: 'ID of a snapshot to restore from',
        required: false,
    }),
};
//# sourceMappingURL=restore.js.map