"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const che_1 = require("../../api/che");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
class Export extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Export);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            yield context_1.ChectlContext.init(flags, this);
            const kube = new kube_1.KubeHelper(flags);
            const cheHelper = new che_1.CheHelper(flags);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Export.id, flags });
            if (!(yield kube.hasReadPermissionsForNamespace(flags.chenamespace))) {
                throw new Error(`E_PERM_DENIED - Permission denied: no read access to '${flags.chenamespace}' namespace`);
            }
            if (!(yield kube.getNamespace(flags.chenamespace))) {
                throw new Error(`E_BAD_NS - Namespace ${flags.chenamespace} does not exist. Please specify it with --chenamespace flag`);
            }
            try {
                const cheCaCert = yield cheHelper.retrieveCheCaCert(flags.chenamespace);
                if (cheCaCert) {
                    const targetFile = yield cheHelper.saveCheCaCert(cheCaCert, flags.destination);
                    this.log(`CodeReady Workspaces self-signed CA certificate is exported to ${targetFile}`);
                }
                else {
                    this.log('Self signed certificate secret not found. Is commonly trusted certificate used?');
                }
            }
            catch (err) {
                this.error(util_1.wrapCommandError(err));
            }
        });
    }
}
exports.default = Export;
Export.description = 'Retrieves CodeReady Workspaces self-signed certificate';
Export.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    destination: flags_1.string({
        char: 'd',
        description: `Destination where to store Che self-signed CA certificate.
                    If the destination is a file (might not exist), then the certificate will be saved there in PEM format.
                    If the destination is a directory, then ${constants_1.DEFAULT_CA_CERT_FILE_NAME} file will be created there with Che certificate in PEM format.
                    If this option is omitted, then Che certificate will be stored in a user's temporary directory as ${constants_1.DEFAULT_CA_CERT_FILE_NAME}.`,
        env: 'CHE_CA_CERT_LOCATION',
        default: '',
    }),
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    telemetry: common_flags_1.CHE_TELEMETRY,
};
//# sourceMappingURL=export.js.map