"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const fs = require("fs");
const che_1 = require("../../api/che");
const che_api_client_1 = require("../../api/che-api-client");
const che_login_manager_1 = require("../../api/che-login-manager");
const context_1 = require("../../api/context");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
class Create extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Create);
            yield context_1.ChectlContext.init(flags, this);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Create.id, flags });
            const devfilePath = this.getDevfilePath(flags.devfile);
            const cheHelper = new che_1.CheHelper(flags);
            const { cheApiEndpoint, accessToken } = yield che_login_manager_1.getLoginData(flags[common_flags_1.CHE_API_ENDPOINT_KEY], flags[common_flags_1.ACCESS_TOKEN_KEY], flags);
            const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            let workspace = yield cheHelper.createWorkspaceFromDevfile(cheApiEndpoint, devfilePath, flags.name, accessToken);
            const workspaceId = workspace.id;
            if (flags.start) {
                yield cheApiClient.startWorkspace(workspaceId, flags.debug, accessToken);
                this.log('Workspace has been successfully created and workspace start request has been sent.');
                this.log('Workspace will be available shortly:');
            }
            else {
                this.log('Workspace has been successfully created:');
            }
            workspace = yield cheApiClient.getWorkspaceById(workspaceId, accessToken);
            if (workspace.links && workspace.links.ide) {
                const workspaceIdeURL = yield cheHelper.buildDashboardURL(workspace.links.ide);
                cli_ux_1.cli.url(workspaceIdeURL, workspaceIdeURL);
            }
            this.exit(0);
        });
    }
    getDevfilePath(devfilePath) {
        if (!devfilePath) {
            if (fs.existsSync('devfile.yaml')) {
                devfilePath = 'devfile.yaml';
            }
            else if (fs.existsSync('devfile.yml')) {
                devfilePath = 'devfile.yml';
            }
            else {
                throw new Error("E_DEVFILE_MISSING - Devfile wasn't specified via '-f' option and 'devfile.yaml' is not present in current directory.");
            }
        }
        return devfilePath;
    }
}
exports.default = Create;
Create.description = 'Creates a workspace from a devfile';
Create.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    devfile: flags_1.string({
        char: 'f',
        description: 'Path or URL to a valid devfile',
        env: 'DEVFILE_PATH',
        required: false,
    }),
    name: flags_1.string({
        description: 'Workspace name: overrides the workspace name to use instead of the one defined in the devfile.',
        required: false,
    }),
    start: flags_1.boolean({
        char: 's',
        description: 'Starts the workspace after creation',
        default: false,
    }),
    debug: flags_1.boolean({
        char: 'd',
        description: 'Debug workspace start. It is useful when workspace start fails and it is needed to print more logs on startup. This flag is used in conjunction with --start flag.',
        default: false,
    }),
    [common_flags_1.CHE_API_ENDPOINT_KEY]: common_flags_1.cheApiEndpoint,
    [common_flags_1.ACCESS_TOKEN_KEY]: common_flags_1.accessToken,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    telemetry: common_flags_1.CHE_TELEMETRY,
};
//# sourceMappingURL=create.js.map