"use strict";
/**
 * Copyright (c) 2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getBackupServerConfiguration = exports.backupServerConfigName = exports.BACKUP_SERVER_CONFIG_CR_NAME_KEY = exports.awsSecretAccessKey = exports.AWS_SECRET_ACCESS_KEY_KEY = exports.awsAccessKeyId = exports.AWS_ACCESS_KEY_ID_KEY = exports.sshKeyFile = exports.SSH_KEY_FILE_KEY = exports.sshKey = exports.SSH_KEY_KEY = exports.backupRestServerPassword = exports.BACKUP_REST_SERVER_PASSWORD_KEY = exports.backupRestServerUsername = exports.BACKUP_REST_SERVER_USERNAME_KEY = exports.backupRepositoryPassword = exports.BACKUP_REPOSITORY_PASSWORD_KEY = exports.backupRepositoryUrl = exports.BACKUP_REPOSITORY_URL_KEY = void 0;
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const fs = require("fs-extra");
const Listr = require("listr");
const constants_1 = require("../../constants");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const backup_restore_1 = require("../../api/backup-restore");
const cli_ux_1 = require("cli-ux");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
exports.BACKUP_REPOSITORY_URL_KEY = 'repository-url';
exports.backupRepositoryUrl = flags_1.string({
    char: 'r',
    description: 'Full address of backup repository. Format is identical to restic.',
    env: 'BACKUP_REPOSITORY_URL',
    required: false,
});
exports.BACKUP_REPOSITORY_PASSWORD_KEY = 'repository-password';
exports.backupRepositoryPassword = flags_1.string({
    char: 'p',
    description: 'Password that is used to encrypt / decrypt backup repository content',
    env: 'BACKUP_REPOSITORY_PASSWORD',
    required: false,
});
exports.BACKUP_REST_SERVER_USERNAME_KEY = 'username';
exports.backupRestServerUsername = flags_1.string({
    description: 'Username for authentication in backup REST server',
    env: 'REST_SERVER_USERNAME',
    required: false,
});
exports.BACKUP_REST_SERVER_PASSWORD_KEY = 'password';
exports.backupRestServerPassword = flags_1.string({
    description: 'Authentication password for backup REST server',
    env: 'REST_SERVER_PASSWORD',
    required: false,
});
exports.SSH_KEY_KEY = 'ssh-key';
exports.sshKey = flags_1.string({
    description: 'Private SSH key for authentication on SFTP server',
    env: 'SSH_KEY',
    required: false,
});
exports.SSH_KEY_FILE_KEY = 'ssh-key-file';
exports.sshKeyFile = flags_1.string({
    description: 'Path to file with private SSH key for authentication on SFTP server',
    env: 'SSH_KEY_FILE',
    required: false,
    exclusive: [exports.SSH_KEY_FILE_KEY],
});
exports.AWS_ACCESS_KEY_ID_KEY = 'aws-access-key-id';
exports.awsAccessKeyId = flags_1.string({
    description: 'AWS access key ID',
    env: 'AWS_ACCESS_KEY_ID',
    required: false,
});
exports.AWS_SECRET_ACCESS_KEY_KEY = 'aws-secret-access-key';
exports.awsSecretAccessKey = flags_1.string({
    description: 'AWS secret access key',
    env: 'AWS_SECRET_ACCESS_KEY',
    required: false,
});
exports.BACKUP_SERVER_CONFIG_CR_NAME_KEY = 'backup-server-config-name';
exports.backupServerConfigName = flags_1.string({
    description: 'Name of custom resource with backup server config',
    env: 'BACKUP_SERVER_CONFIG_NAME',
    required: false,
    exclusive: [exports.BACKUP_REPOSITORY_URL_KEY, exports.BACKUP_REPOSITORY_PASSWORD_KEY],
});
class Backup extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Backup);
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Backup.id, flags });
            const tasks = new Listr([], ctx.listrOptions);
            const apiTasks = new api_1.ApiTasks();
            tasks.add(apiTasks.testApiTasks(flags));
            tasks.add(this.getBackupTasks(flags));
            try {
                yield tasks.run(ctx);
            }
            catch (err) {
                this.error(util_1.wrapCommandError(err));
            }
            if (ctx.snapshotId) {
                cli_ux_1.cli.info(`Backup snapshot ID: ${ctx.snapshotId}`);
            }
            cli_ux_1.cli.info(util_1.getCommandSuccessMessage());
            util_1.notifyCommandCompletedSuccessfully();
        });
    }
    getBackupTasks(flags) {
        return [
            {
                title: 'Scheduling backup...',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const backupServerConfig = getBackupServerConfiguration(flags);
                    yield backup_restore_1.requestBackup(flags.chenamespace, backupServerConfig);
                    task.title = `${task.title}OK`;
                }),
            },
            {
                title: 'Waiting until backup process finishes...',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const kube = new kube_1.KubeHelper(flags);
                    let backupStatus = {};
                    do {
                        yield cli_ux_1.cli.wait(1000);
                        const backupCr = yield kube.getCustomResource(flags.chenamespace, constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_CLUSTER_BACKUP_KIND_PLURAL);
                        if (!backupCr.status) {
                            continue;
                        }
                        backupStatus = backupCr.status;
                        if (backupStatus.stage) {
                            task.title = `Waiting until backup process finishes: ${backupStatus.stage}`;
                        }
                    } while (backupStatus.state === 'InProgress');
                    if (backupStatus.state === 'Failed') {
                        throw new Error(`Failed to create backup: ${backupStatus.message}`);
                    }
                    ctx.snapshotId = backupStatus.snapshotId;
                    task.title = 'Waiting until backup process finishes...OK';
                }),
            },
        ];
    }
}
exports.default = Backup;
Backup.description = 'Backup CodeReady Workspaces installation';
Backup.examples = [
    '# Reuse existing backup configuration or create and use internal backup server if none exists:\n' +
        'crwctl server:backup',
    '# Create and use configuration for REST backup server:\n' +
        'crwctl server:backup -r rest:http://my-sert-server.net:4000/che-backup -p repopassword',
    '# Create and use configuration for AWS S3 (and API compatible) backup server (bucket should be precreated):\n' +
        'crwctl server:backup -r s3:s3.amazonaws.com/bucketche -p repopassword',
    '# Create and use configuration for SFTP backup server:\n' +
        'crwctl server:backup -r sftp:user@my-server.net:/srv/sftp/che-data -p repopassword',
];
Backup.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    [exports.BACKUP_REPOSITORY_URL_KEY]: exports.backupRepositoryUrl,
    [exports.BACKUP_REPOSITORY_PASSWORD_KEY]: exports.backupRepositoryPassword,
    [exports.BACKUP_REST_SERVER_USERNAME_KEY]: exports.backupRestServerUsername,
    [exports.BACKUP_REST_SERVER_PASSWORD_KEY]: exports.backupRestServerPassword,
    [exports.SSH_KEY_KEY]: exports.sshKey,
    [exports.SSH_KEY_FILE_KEY]: exports.sshKeyFile,
    [exports.AWS_ACCESS_KEY_ID_KEY]: exports.awsAccessKeyId,
    [exports.AWS_SECRET_ACCESS_KEY_KEY]: exports.awsSecretAccessKey,
    [exports.BACKUP_SERVER_CONFIG_CR_NAME_KEY]: exports.backupServerConfigName,
};
function getBackupServerConfiguration(flags) {
    if (flags[exports.BACKUP_SERVER_CONFIG_CR_NAME_KEY]) {
        // Backup server configuration CR name is provided
        return flags[exports.BACKUP_SERVER_CONFIG_CR_NAME_KEY];
    }
    const repoUrl = flags[exports.BACKUP_REPOSITORY_URL_KEY];
    const repoPassword = flags[exports.BACKUP_REPOSITORY_PASSWORD_KEY];
    if (!repoUrl && !repoPassword) {
        // If there is no repo url and password, it is supposed that command do not have any backup server configured.
        // This could mean:
        // 1. Reuse last backup configuration, if any
        // 2. If no previous configuration, set up and use internal backup server
        return;
    }
    // Ensure both repoUrl and repoPassword are set
    if (!repoUrl) {
        throw new Error(`Parameter ${exports.BACKUP_REPOSITORY_URL_KEY} required`);
    }
    if (!repoPassword) {
        throw new Error(`Parameter ${exports.BACKUP_REPOSITORY_PASSWORD_KEY} required`);
    }
    const serverType = backup_restore_1.getBackupServerType(repoUrl);
    const backupServerConfig = {
        type: serverType,
        url: repoUrl,
        repoPassword,
    };
    switch (serverType) {
        case 'rest':
            const username = flags[exports.BACKUP_REST_SERVER_USERNAME_KEY];
            const password = flags[exports.BACKUP_REST_SERVER_PASSWORD_KEY];
            if (username && password) {
                backupServerConfig.credentials = { username, password };
            }
            else if (username || password) {
                // Only one parameter given, but this is not allowed
                if (username) {
                    throw new Error(`${exports.BACKUP_REST_SERVER_PASSWORD_KEY} parameter should be provided`);
                }
                else {
                    throw new Error(`${exports.BACKUP_REST_SERVER_USERNAME_KEY} parameter should be provided`);
                }
            }
            // If both username and password are empty, then authentication on the server is turned off
            warnAboutIgnoredCredentialsParameters(flags, [exports.BACKUP_REST_SERVER_USERNAME_KEY, exports.BACKUP_REST_SERVER_PASSWORD_KEY]);
            break;
        case 'sftp':
            if (!flags[exports.SSH_KEY_KEY] && !flags[exports.SSH_KEY_FILE_KEY]) {
                throw new Error(`SSH key should be provided via ${exports.SSH_KEY_KEY} or ${exports.SSH_KEY_FILE_KEY}`);
            }
            else if (flags[exports.SSH_KEY_KEY] && flags[exports.SSH_KEY_FILE_KEY]) {
                throw new Error(`Only one of ${exports.SSH_KEY_KEY} and ${exports.SSH_KEY_FILE_KEY} parameters should be provided`);
            }
            let sshKey;
            if (flags[exports.SSH_KEY_KEY]) {
                sshKey = flags[exports.SSH_KEY_KEY];
            }
            else {
                // Read SSH key from file
                const sshKeyFilePath = flags[exports.SSH_KEY_FILE_KEY];
                if (!fs.existsSync(sshKeyFilePath)) {
                    throw new Error(`File ${sshKeyFilePath} with SSH key doesn't exist`);
                }
                sshKey = fs.readFileSync(sshKeyFilePath).toString().trim();
            }
            backupServerConfig.credentials = { sshKey };
            warnAboutIgnoredCredentialsParameters(flags, [exports.SSH_KEY_KEY, exports.SSH_KEY_FILE_KEY]);
            break;
        case 's3':
            if (!flags[exports.AWS_ACCESS_KEY_ID_KEY]) {
                throw new Error(`${exports.AWS_ACCESS_KEY_ID_KEY} should be provided`);
            }
            if (!flags[exports.AWS_SECRET_ACCESS_KEY_KEY]) {
                throw new Error(`${exports.AWS_SECRET_ACCESS_KEY_KEY} should be provided`);
            }
            backupServerConfig.credentials = {
                awsAccessKeyId: flags[exports.AWS_ACCESS_KEY_ID_KEY],
                awsSecretAccessKey: flags[exports.AWS_SECRET_ACCESS_KEY_KEY],
            };
            warnAboutIgnoredCredentialsParameters(flags, [exports.AWS_ACCESS_KEY_ID_KEY, exports.AWS_SECRET_ACCESS_KEY_KEY]);
            break;
        default:
            throw new Error(`Unrecognized backup server protocol in '${repoUrl}' url`);
    }
    return backupServerConfig;
}
exports.getBackupServerConfiguration = getBackupServerConfiguration;
function warnAboutIgnoredCredentialsParameters(flags, requiredParameters = []) {
    const credentialsParametersKeys = [
        exports.BACKUP_REST_SERVER_USERNAME_KEY,
        exports.BACKUP_REST_SERVER_PASSWORD_KEY,
        exports.SSH_KEY_KEY,
        exports.SSH_KEY_FILE_KEY,
        exports.AWS_ACCESS_KEY_ID_KEY,
        exports.AWS_SECRET_ACCESS_KEY_KEY,
    ];
    for (const key of credentialsParametersKeys) {
        if (flags[key] && !requiredParameters.includes(key)) {
            cli_ux_1.cli.warn(`${key} parameter is ignored`);
        }
    }
}
//# sourceMappingURL=backup.js.map