/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
export interface LoginData {
    cheApiEndpoint: string;
    accessToken: string | undefined;
}
export interface CheServerLoginConfig {
    version?: string;
    lastLoginUrl?: string;
    lastUserName?: string;
    logins?: Logins;
}
export declare type Logins = {
    [key: string]: ServerLogins;
};
export declare type ServerLogins = {
    [key: string]: RefreshTokenLoginRecord;
};
export declare type LoginRecord = RefreshTokenLoginRecord | PasswordLoginRecord | OcUserTokenLoginRecord;
export interface RefreshTokenLoginRecord {
    refreshToken: string;
    expires: number;
}
export interface OcUserTokenLoginRecord {
    subjectToken: string;
    subjectIssuer: string;
}
export interface PasswordLoginRecord {
    username: string;
    password: string;
}
export declare function isRefreshTokenLoginData(loginData: LoginRecord): loginData is RefreshTokenLoginRecord;
export declare function isOcUserTokenLoginData(loginData: LoginRecord): loginData is OcUserTokenLoginRecord;
export declare function isPasswordLoginData(loginData: LoginRecord): loginData is PasswordLoginRecord;
/**
 * Che server login sessions manager. Singleton.
 * Uses refresh tokens for authentication.
 * Usually, just using of getLoginData function is suitable.
 */
export declare class CheServerLoginManager {
    private loginData;
    private apiUrl;
    private username;
    private readonly dataFilePath;
    private readonly axios;
    private constructor();
    /**
     * Returns Che server login sessions manager.
     */
    static getInstance(): Promise<CheServerLoginManager>;
    /**
     * Checks whether login credentials exists for given server and user.
     * @param apiUrl API URL of the Che server
     * @param username username
     */
    hasLoginFor(apiUrl: string, username?: string): boolean;
    getCurrentLoginInfo(): {
        cheApiEndpoint: string;
        username: string;
    };
    getCurrentServerApiUrl(): string;
    getAllLogins(): Map<string, string[]>;
    /**
     * Logins user in specified instance of Che Server.
     * Makes this login data default context.
     * If a context with the same data already exists it will be replaced.
     * If provided data is invalid, exception will be thrown.
     * Returns username of the login.
     * @param apiUrl Che server API URL
     * @param loginRecord user credentials
     */
    setLoginContext(apiUrl: string, loginRecord: LoginRecord): Promise<string>;
    /**
     * Changes current login.
     */
    switchLoginContext(apiUrl: string, username: string): Promise<void>;
    /**
     * Logouts user from specified Che server.
     * If no parameters given current login session will be deleted.
     * @param apiUrl Che server API URL
     * @param username username on the given server
     */
    deleteLoginContext(apiUrl?: string, username?: string): void;
    private readLoginData;
    private saveLoginData;
    /**
     * Searches for login data by API URL and user name.
     * Returns undefined if nothing found by given keys.
     */
    private getLoginRecord;
    /**
     * Sets current login credentials by given API URL and username.
     * If loginRecord is provided, then a new credentials are added, replacing existing if any.
     * This method doesn't check credentials validity.
     * Returns true if operation was successful.
     */
    private setCurrentLoginContext;
    private removeExpiredLogins;
    private retrieveKeycloakSettings;
    /**
     * Returns new Keycloak access token for current login session.
     * Updates session timeout.
     */
    getNewAccessToken(): Promise<string>;
    private keycloakAuth;
    private getKeycloakAuthDataByUserNameAndPassword;
    private getKeycloakAuthDataByRefreshToken;
    private getKeycloakAuthDataByOcToken;
    private requestKeycloakAuth;
    private getCurrentUserName;
}
/**
 * Helper function to get valid credentials. Designed to be used from commands.
 * @param cheApiEndpoint user provided server API URL if any
 * @param accessToken user provied access token if any
 */
export declare function getLoginData(cheApiEndpoint: string, accessToken: string | undefined, flags: any): Promise<LoginData>;
/**
 * Gets cheApiEndpoint for the given namespace.
 */
export declare function getCheApiEndpoint(flags: any): Promise<string>;
