/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.server.core.schema;


import static org.apache.directory.server.core.integ.IntegrationUtils.getSchemaContext;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import javax.naming.InvalidNameException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;
import javax.naming.OperationNotSupportedException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;

import org.apache.directory.server.core.integ.FrameworkRunner;
import org.apache.directory.shared.ldap.exception.LdapException;
import org.apache.directory.shared.ldap.ldif.LdifUtils;
import org.apache.directory.shared.ldap.name.DN;
import org.apache.directory.shared.ldap.schema.ObjectClass;
import org.apache.directory.shared.ldap.schema.registries.ObjectClassRegistry;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;


/**
 * A test case which tests the addition of various schema elements
 * to the ldap server.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 * @version $Rev$
 */
@RunWith(FrameworkRunner.class)
public class MetaObjectClassHandlerIT extends AbstractMetaSchemaObjectHandler
{
    private static final String NAME = "testObjectClass";
    private static final String NEW_NAME = "alternateName";
    private static final String DEPENDEE_NAME = "dependeeName";

    private static final String DESCRIPTION0 = "A test objectClass";
    private static final String DESCRIPTION1 = "An alternate description";
    
    private static final String OID = "1.3.6.1.4.1.18060.0.4.0.3.100000";
    private static final String NEW_OID = "1.3.6.1.4.1.18060.0.4.0.3.100001";
    private static final String DEPENDEE_OID = "1.3.6.1.4.1.18060.0.4.0.3.100002";


    private static ObjectClassRegistry getObjectClassRegistry()
    {
        return service.getSchemaManager().getObjectClassRegistry();
    }
    
    
    private DN addObjectClass() throws Exception
    {
        Attributes attrs = LdifUtils.createAttributes( 
            "objectClass: top",
            "objectClass: metaTop",
            "objectClass: metaObjectClass",
            "m-oid: " + OID,
            "m-name: " + NAME,
            "m-description: " + DESCRIPTION0,
            "m-typeObjectClass: AUXILIARY",
            "m-must: cn",
            "m-may: ou" );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        return dn;
    }

    
    // ----------------------------------------------------------------------
    // Test all core methods with normal operational pathways
    // ----------------------------------------------------------------------
    @Test
    public void testAddObjectClassToEnabledSchema() throws Exception
    {
        DN dn = addObjectClass();
        
        assertTrue( getObjectClassRegistry().contains( OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( OID ), "apachemeta" );
        assertTrue( isOnDisk( dn ) );
    }
    
    
    @Test
    public void testAddObjectClassToDisabledSchema() throws Exception
    {
        DN dn = addObjectClassToDisabledSchema();
        
        assertFalse( "adding new objectClass to disabled schema should not register it into the registries", 
            getObjectClassRegistry().contains( OID ) );
        assertTrue( isOnDisk( dn ) );
    }


    @Test
    public void testAddObjectClassToUnloadedSchema() throws Exception
    {
        Attributes attrs = LdifUtils.createAttributes( 
            "objectClass: top",
            "objectClass: metaTop",
            "objectClass: metaObjectClass",
            "m-oid: " + OID,
            "m-name: " + NAME,
            "m-description: " + DESCRIPTION0,
            "m-typeObjectClass: AUXILIARY",
            "m-must: cn",
            "m-may: ou" );

        DN dn = getObjectClassContainer( "notloaded" );
        dn.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
            fail( "Should not be there" );
        }
        catch( NameNotFoundException nnfe )
        {
            // Excpected result
        }
        
        assertFalse( "adding new objectClass to disabled schema should not register it into the registries", 
            getObjectClassRegistry().contains( OID ) );
        assertFalse( isOnDisk( dn ) );
    }


    @Test
    public void testDeleteObjectClassFromEnabledSchema() throws Exception
    {
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        addObjectClass();

        assertTrue( "objectClass should be removed from the registry after being deleted", 
            getObjectClassRegistry().contains( OID ) );
        assertTrue( isOnDisk( dn ) );

        getSchemaContext( service ).destroySubcontext( DN.toName( dn ) );

        assertFalse( "objectClass should be removed from the registry after being deleted", 
            getObjectClassRegistry().contains( OID ) );

        try
        {
            getObjectClassRegistry().lookup( OID );
            fail( "objectClass lookup should fail after deleting it" );
        }
        catch( LdapException e )
        {
        }

        assertFalse( isOnDisk( dn ) );
    }

    
    @Test
    public void testDeleteObjectClassFromDisabledSchema() throws Exception
    {
        DN dn = getObjectClassContainer( "nis" );
        dn.add( "m-oid" + "=" + OID );
        addObjectClassToDisabledSchema();
        
        assertFalse( "objectClass should be removed from the registry after being deleted", 
            getObjectClassRegistry().contains( OID ) );
        assertTrue( isOnDisk( dn ) );
        
        getSchemaContext( service ).destroySubcontext( DN.toName( dn ) );

        assertFalse( "objectClass should be removed from the registry after being deleted", 
            getObjectClassRegistry().contains( OID ) );

        try
        {
            getObjectClassRegistry().lookup( OID );
            fail( "objectClass lookup should fail after deleting it" );
        }
        catch( LdapException e )
        {
        }
        
        assertFalse( isOnDisk( dn ) );
    }
    

    @Test
    @Ignore
    public void testRenameObjectClassType() throws Exception
    {
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        addObjectClass();
        
        DN newdn = getObjectClassContainer( "apachemeta" );
        newdn.add( "m-oid" + "=" + NEW_OID );
        getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );

        assertFalse( "old objectClass OID should be removed from the registry after being renamed", 
            getObjectClassRegistry().contains( OID ) );

        //noinspection EmptyCatchBlock
        try
        {
            getObjectClassRegistry().lookup( OID );
            fail( "objectClass lookup should fail after renaming the objectClass" );
        }
        catch( LdapException e )
        {
        }

        assertTrue( getObjectClassRegistry().contains( NEW_OID ) );
    }


    @Test
    @Ignore
    public void testMoveObjectClass() throws Exception
    {
        addObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        DN newdn = getObjectClassContainer( "apache" );
        newdn.add( "m-oid" + "=" + OID );
        
        getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );

        assertTrue( "objectClass OID should still be present", 
            getObjectClassRegistry().contains( OID ) );
        
        assertEquals( "objectClass schema should be set to apache not apachemeta", 
            getObjectClassRegistry().getSchemaName( OID ), "apache" );
    }


    @Test
    @Ignore
    public void testMoveObjectClassAndChangeRdn() throws Exception
    {
        addObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        DN newdn = getObjectClassContainer( "apache" );
        newdn.add( "m-oid" + "=" + NEW_OID );
        
        getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );

        assertFalse( "old objectClass OID should NOT be present", 
            getObjectClassRegistry().contains( OID ) );
        
        assertTrue( "new objectClass OID should be present", 
            getObjectClassRegistry().contains( NEW_OID ) );
        
        assertEquals( "objectClass with new oid should have schema set to apache NOT apachemeta", 
            getObjectClassRegistry().getSchemaName( NEW_OID ), "apache" );
    }

    
    @Test
    @Ignore
    public void testModifyObjectClassWithModificationItems() throws Exception
    {
        addObjectClass();
        
        ObjectClass oc = getObjectClassRegistry().lookup( OID );
        assertEquals( oc.getDescription(), DESCRIPTION0 );
        assertEquals( oc.getName(), NAME );

        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        
        ModificationItem[] mods = new ModificationItem[2];
        Attribute attr = new BasicAttribute( "m-description", DESCRIPTION1 );
        mods[0] = new ModificationItem( DirContext.REPLACE_ATTRIBUTE, attr );
        attr = new BasicAttribute( "m-name", NEW_NAME );
        mods[1] = new ModificationItem( DirContext.REPLACE_ATTRIBUTE, attr );
        getSchemaContext( service ).modifyAttributes( DN.toName( dn ), mods );

        assertTrue( "objectClass OID should still be present", 
            getObjectClassRegistry().contains( OID ) );
        
        assertEquals( "objectClass schema should be set to apachemeta", 
            getObjectClassRegistry().getSchemaName( OID ), "apachemeta" );
        
        oc = getObjectClassRegistry().lookup( OID );
        assertEquals( oc.getDescription(), DESCRIPTION1 );
        assertEquals( oc.getName(), NEW_NAME );
    }

    
    @Test
    @Ignore
    public void testModifyObjectClassWithAttributes() throws Exception
    {
        addObjectClass();
        
        ObjectClass oc = getObjectClassRegistry().lookup( OID );
        assertEquals( oc.getDescription(), DESCRIPTION0 );
        assertEquals( oc.getName(), NAME );

        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        
        Attributes mods = new BasicAttributes( true );
        mods.put( "m-description", DESCRIPTION1 );
        mods.put( "m-name", NEW_NAME );
        getSchemaContext( service ).modifyAttributes( DN.toName( dn ), DirContext.REPLACE_ATTRIBUTE, mods );

        assertTrue( "objectClass OID should still be present", 
            getObjectClassRegistry().contains( OID ) );
        
        assertEquals( "objectClass schema should be set to apachemeta", 
            getObjectClassRegistry().getSchemaName( OID ), "apachemeta" );

        oc = getObjectClassRegistry().lookup( OID );
        assertEquals( oc.getDescription(), DESCRIPTION1 );
        assertEquals( oc.getName(), NEW_NAME );
    }
    

    // ----------------------------------------------------------------------
    // Test move, rename, and delete when a OC exists and uses the OC as sup
    // ----------------------------------------------------------------------
    private void addDependeeObjectClass() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        attrs.put( "m-oid", DEPENDEE_OID );
        attrs.put( "m-name", DEPENDEE_NAME );
        attrs.put( "m-description", DESCRIPTION0 );
        attrs.put( "m-typeObjectClass", "AUXILIARY" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        attrs.put( "m-supObjectClass", OID );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + DEPENDEE_OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        assertTrue( getObjectClassRegistry().contains( DEPENDEE_OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( DEPENDEE_OID ), "apachemeta" );
    }

    
    @Test
    public void testDeleteObjectClassWhenInUse() throws Exception
    {
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        addObjectClass();
        addDependeeObjectClass();
        
        try
        {
            getSchemaContext( service ).destroySubcontext( DN.toName( dn ) );
            fail( "should not be able to delete a objectClass in use" );
        }
        catch( OperationNotSupportedException e ) 
        {
        }

        assertTrue( "objectClass should still be in the registry after delete failure", 
            getObjectClassRegistry().contains( OID ) );
    }
    
    
    @Test
    @Ignore
    public void testMoveObjectClassWhenInUse() throws Exception
    {
        addObjectClass();
        addDependeeObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        DN newdn = getObjectClassContainer( "apache" );
        newdn.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );
            fail( "should not be able to move a objectClass in use" );
        }
        catch( OperationNotSupportedException e ) 
        {
        }

        assertTrue( "objectClass should still be in the registry after move failure", 
            getObjectClassRegistry().contains( OID ) );
    }


    @Test
    @Ignore
    public void testMoveObjectClassAndChangeRdnWhenInUse() throws Exception
    {
        addObjectClass();
        addDependeeObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        DN newdn = getObjectClassContainer( "apache" );
        newdn.add( "m-oid" + "=" + NEW_OID );
        
        try
        {
            getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );
            fail( "should not be able to move an objectClass in use" );
        }
        catch( OperationNotSupportedException e ) 
        {
        }

        assertTrue( "ObjectClass should still be in the registry after move failure", 
            getObjectClassRegistry().contains( OID ) );
    }

    
    @Test
    @Ignore
    public void testRenameObjectClassWhenInUse() throws Exception
    {
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        addObjectClass();
        addDependeeObjectClass();
        
        DN newdn = getObjectClassContainer( "apachemeta" );
        newdn.add( "m-oid" + "=" + NEW_OID );
        
        try
        {
            getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );
            fail( "should not be able to rename an objectClass in use" );
        }
        catch( OperationNotSupportedException e ) 
        {
        }

        assertTrue( "objectClass should still be in the registry after rename failure", 
            getObjectClassRegistry().contains( OID ) );
    }


    // ----------------------------------------------------------------------
    // Let's try some freaky stuff
    // ----------------------------------------------------------------------
    @Test
    @Ignore
    public void testMoveObjectClassToTop() throws Exception
    {
        addObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        DN top = new DN();
        top.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( top ) );
            fail( "should not be able to move a objectClass up to ou=schema" );
        }
        catch( InvalidNameException e ) 
        {
        }

        assertTrue( "objectClass should still be in the registry after move failure", 
            getObjectClassRegistry().contains( OID ) );
    }


    @Test
    @Ignore
    public void testMoveObjectClassToComparatorContainer() throws Exception
    {
        addObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        DN newdn = new DN( "ou=comparators,cn=apachemeta" );
        newdn.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );
            fail( "should not be able to move a objectClass into comparators container" );
        }
        catch( InvalidNameException e ) 
        {
        }

        assertTrue( "objectClass should still be in the registry after move failure", 
            getObjectClassRegistry().contains( OID ) );
    }

    
    private DN addObjectClassToDisabledSchema() throws Exception
    {
        Attributes attrs = LdifUtils.createAttributes( 
            "objectClass: top",
            "objectClass: metaTop",
            "objectClass: metaObjectClass",
            "m-oid: " + OID,
            "m-name: " + NAME,
            "m-description: " + DESCRIPTION0,
            "m-typeObjectClass: AUXILIARY",
            "m-must: cn",
            "m-may: ou" );

        DN dn = getObjectClassContainer( "nis" );
        dn.add( "m-oid" + "=" + OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        return dn;
    }
    
    
    @Test
    @Ignore
    public void testMoveObjectClassToDisabledSchema() throws Exception
    {
        addObjectClass();
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );

        // nis is inactive by default
        DN newdn = getObjectClassContainer( "nis" );
        newdn.add( "m-oid" + "=" + OID );
        
        getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );

        assertFalse( "objectClass OID should no longer be present", 
            getObjectClassRegistry().contains( OID ) );
    }


    @Test
    @Ignore
    public void testMoveObjectClassToEnabledSchema() throws Exception
    {
        addObjectClassToDisabledSchema();
        
        // nis is inactive by default
        DN dn = getObjectClassContainer( "nis" );
        dn.add( "m-oid" + "=" + OID );

        assertFalse( "objectClass OID should NOT be present when added to disabled nis schema", 
            getObjectClassRegistry().contains( OID ) );

        DN newdn = getObjectClassContainer( "apachemeta" );
        newdn.add( "m-oid" + "=" + OID );
        
        getSchemaContext( service ).rename( DN.toName( dn ), DN.toName( newdn ) );

        assertTrue( "objectClass OID should be present when moved to enabled schema", 
            getObjectClassRegistry().contains( OID ) );
        
        assertEquals( "objectClass should be in apachemeta schema after move", 
            getObjectClassRegistry().getSchemaName( OID ), "apachemeta" );
    }
    
    // ----------------------------------------------------------------------
    // Let's test the Abstract, Auiliary and Structural inheritence enforcement
    // ----------------------------------------------------------------------
    /**
     * Check that we can create an ABSTRACT OC which inherit from an ABSTRACT OC
     */
    @Test
    public void testAddAbstractOCinheritingFromAbstractOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "ABSTRACT" );
        attrs.put( "m-supObjectClass", "top" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        assertTrue( getObjectClassRegistry().contains( OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( OID ), "apachemeta" );
    }

    
    /**
     * Check that we can't create an ABSTRACT OC which inherit from an AUXILIARY OC
     */
    @Test
    public void testAddAbstractOCinheritingFromAuxiliaryOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "ABSTRACT" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        Attribute sup = new BasicAttribute( "m-supObjectClass" );
        sup.add( "top" );
        sup.add( "javaSerializedObject");
        attrs.put( sup );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
            fail();
        }
        catch ( NamingException ne )
        {
            assertTrue( true );
        }
    }
    
    
    /**
     * Check that we can't create an ABSTRACT OC which inherit from an STRUCTURAL OC
     */
    @Test
    public void testAddAbstractOCinheritingFromStructuralOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "ABSTRACT" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        Attribute sup = new BasicAttribute( "m-supObjectClass" );
        sup.add( "top" );
        sup.add( "person");
        attrs.put( sup );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
            fail();
        }
        catch ( NamingException ne )
        {
            assertTrue( true );
        }
    }
    
    
    /**
     * Check that we can create an AUXILIARY OC which inherit from an ABSTRACT OC
     */
    @Test
    public void testAddAuxiliaryOCinheritingFromAbstractOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", NEW_OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "AUXILIARY" );
        attrs.put( "m-supObjectClass", "top" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + NEW_OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        assertTrue( getObjectClassRegistry().contains( NEW_OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( NEW_OID ), "apachemeta" );
    }

    
    /**
     * Check that we can create an AUXILIARY OC which inherit from an AUXILIARY OC
     */
    @Test
    public void testAddAuxiliaryOCinheritingFromAuxiliaryOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", NEW_OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "AUXILIARY" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        Attribute sup = new BasicAttribute( "m-supObjectClass" );
        sup.add( "top" );
        sup.add( "javaNamingReference");
        attrs.put( sup );

        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + NEW_OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        assertTrue( getObjectClassRegistry().contains( NEW_OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( NEW_OID ), "apachemeta" );
    }

    
    /**
     * Check that we can't create an Auxiliary OC which inherit from an STRUCTURAL OC
     */
    @Test
    public void testAddAuxiliaryOCinheritingFromStructuralOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "ABSTRACT" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        Attribute sup = new BasicAttribute( "m-supObjectClass" );
        sup.add( "top" );
        sup.add( "person");
        attrs.put( sup );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + OID );
        
        try
        {
            getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
            fail();
        }
        catch ( NamingException ne )
        {
            assertTrue( true );
        }
    }

    
    /**
     * Check that we can create a STRUCTURAL OC which inherit from an ABSTRACT OC
     */
    @Test
    public void testAddStructuralOCinheritingFromAbstractOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", NEW_OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "STRUCTURAL" );
        attrs.put( "m-supObjectClass", "top" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + NEW_OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        assertTrue( getObjectClassRegistry().contains( NEW_OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( NEW_OID ), "apachemeta" );
    }

    
    /**
     * Check that we can create a STRUCTURAL OC which inherit from an AUXILIARY OC
     */
    @Test
    public void testAddStructuralOCinheritingFromAuxiliaryOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", NEW_OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "STRUCTURAL" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        Attribute sup = new BasicAttribute( "m-supObjectClass" );
        sup.add( "top" );
        sup.add( "javaNamingReference");
        attrs.put( sup );

        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + NEW_OID );
        
        try
        {
            getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
            fail();
        }
        catch ( NamingException ne )
        {
            assertTrue( true );
        }
    }

    
    /**
     * Check that we can create a STRUCTURAL OC which inherit from an STRUCTURAL OC
     */
    @Test
    public void testAddStructuralOCinheritingFromStructuralOC() throws Exception
    {
        Attributes attrs = new BasicAttributes( true );
        Attribute oc = new BasicAttribute( "objectClass", "top" );
        oc.add( "metaTop" );
        oc.add( "metaObjectClass" );
        attrs.put( oc );
        
        attrs.put( "m-oid", NEW_OID );
        attrs.put( "m-name", "abstractOCtest");
        attrs.put( "m-description", "An abstract oC inheriting from top" );
        attrs.put( "m-typeObjectClass", "STRUCTURAL" );
        attrs.put( "m-must", "cn" );
        attrs.put( "m-may", "ou" );
        
        Attribute sup = new BasicAttribute( "m-supObjectClass" );
        sup.add( "top" );
        sup.add( "person");
        attrs.put( sup );

        DN dn = getObjectClassContainer( "apachemeta" );
        dn.add( "m-oid" + "=" + NEW_OID );
        getSchemaContext( service ).createSubcontext( DN.toName( dn ), attrs );
        
        assertTrue( getObjectClassRegistry().contains( NEW_OID ) );
        assertEquals( getObjectClassRegistry().getSchemaName( NEW_OID ), "apachemeta" );
    }
}
